--[[
Script to animate some indoor parts with gps mod and enhanced vehicle mod

Author:		Ifko[nator]
Date:		29.06.2020
Version:	1.0

History:	V 1.0 @ 29.06.2020 - initial release in FS 19

XML entry example: (You don't must use all of them, same drive symbols can be used multiply times!)

You can for every drive symbol, except 'CONTROL_LAMP', use: (like i did in the example for 'FOUR_WHEEL')

'baseColor' --## default value is 'GREY' (support all default dashboard colors, brand colors and color strings like '1 1 1 0')
'emitColor' --## default value is 'GREEN' (support all default dashboard colors, brand colors and color strings like '1 1 1 0')
'intensity' --## default value is '1'

	<animatedIndoorParts>
		<fillLevelBar node="fillLevelBar" scaleDirection="X"/>
		<fillLevelBar node="fillLevelBar2" scaleDirection="ALL"/>
		
		<driveSymbols>
			<driveSymbol name="FOUR_WHEEL" node="fourWheel" baseColor="GREY" emitColor="GREEN" intensity="1"/>
			<driveSymbol name="DIFF_LOCK_FRONT" node="diffLockFront"/>
			<driveSymbol name="DIFF_LOCK_BACK" node="diffLockBack"/>
			<driveSymbol name="PTO_FRONT" node="ptoFront"/>
			<driveSymbol name="PTO_BACK" node="ptoBack"/>
			<driveSymbol name="HANDBRAKE" node="handBrake"/>
			<driveSymbol name="TIPPING_SIDE_LEFT_1" node="tippingSideLeft1"/>
			<driveSymbol name="TIPPING_SIDE_RIGHT_1" node="tippingSideRight1"/>
			<driveSymbol name="TIPPING_SIDE_BACK_1" node="tippingSideBack1"/>
			<driveSymbol name="TIPPING_SIDE_GRAINDOOR_1" node="tippingSideGrainDoor1"/>
			<driveSymbol name="TIPPING_SIDE_LEFT_2" node="tippingSideLeft2"/>
			<driveSymbol name="TIPPING_SIDE_RIGHT_2" node="tippingSideRight2"/>
			<driveSymbol name="TIPPING_SIDE_BACK_2" node="tippingSideBack2"/>
			<driveSymbol name="TIPPING_SIDE_GRAINDOOR_2" node="tippingSideGrainDoor2"/>
			<driveSymbol name="TIPPING_ACTIVE" node="tippingActive"/>
			<driveSymbol name="TIPPING_INACTIVE" node="tippingInActive"/>
			<driveSymbol name="TRAILER_1" node="trailer1"/>
			<driveSymbol name="TRAILER_2" node="trailer2"/>
			<driveSymbol name="TOOL_FRONT_LOWERED" node="toolFrontLowered"/>
			<driveSymbol name="TOOL_FRONT_LIFTED" node="toolFrontLifted"/>
			<driveSymbol name="TOOL_BACK_LOWERED" node="toolBackLowered"/>
			<driveSymbol name="TOOL_BACK_LIFTED" node="toolBackLifted"/>
			<driveSymbol name="CONTROL_LAMP" node="controlLamp" displayTime="2200"/>
			<driveSymbol name="GPS_ACTIVE" node="gpsActive" />
			<driveSymbol name="GPS_STEERING_ACTIVE" node="gpsSteeringActive"/>
			<driveSymbol name="GPS_LANE_PLUS" node="gpsLanePlus"/>
			<driveSymbol name="GPS_LANE_MINUS" node="gpsLaneMinus"/>
			<driveSymbol name="STEERAXLE_IS_LOCKED" node="steerAxleIsLocked"/>
			<driveSymbol name="STEERAXLE_IS_UNLOCKED" node="steerAxleIsUnLocked"/>
			<driveSymbol name="RIDGE_MARKER_LEFT" node="ridgeMarkerLeft"/>
			<driveSymbol name="RIDGE_MARKER_RIGHT" node="ridgeMarkerRight"/>
			<driveSymbol name="TOOL_FRONT_IS_UNFOLDED" node="toolFrontIsUnFolded"/>
			<driveSymbol name="TOOL_FRONT_IS_FOLDED" node="toolFrontIsFolded"/>
			<driveSymbol name="TOOL_BACK_IS_UNFOLDED" node="toolBackIsUnFolded"/>
			<driveSymbol name="TOOL_BACK_IS_FOLDED" node="toolBackIsFolded"/>
			<driveSymbol name="TIPPING_ACTIVE_TRUCK" node="tippingTruckActive"/>
			<driveSymbol name="ABS_TRUCK" node="absTruck"/>
			<driveSymbol name="ABS_TRAILER" node="absTrailer"/>
			<driveSymbol name="TIPPING_ACTIVE_TRAILER_1" node="tippingActiveTrailer1"/>
			<driveSymbol name="TIPPING_ACTIVE_TRAILER_2" node="tippingActiveTrailer2"/>
			<driveSymbol name="WARNING_BRAKE_COMPRESSOR_FILL_LEVEL_TO_LOW" node="warningBrakeCompressorFillLevelToLow"/> <!--active when air compressor is refilling-->
			<driveSymbol name="WARNING_WORKSHOP_1" node="warningWorkshop1"/> <!--active when vehicle damage >= 70%-->
			<driveSymbol name="WARNING_WORKSHOP_2" node="warningWorkshop2"/> <!--active when vehicle damage >= 80%-->
			<driveSymbol name="WEARED_OUT_BRAKES" node="wearedOutBrakes"/> <!--active when vehicle damage >= 82%-->
			<driveSymbol name="MOTOR_NEED_MAINTRACE" node="motorNeedMaintrace"/> <!--active when vehicle damage >= 90%-->
			<driveSymbol name="LOW_OIL_PRESSURE" node="lowOilPressure"/> <!--active when vehicle damage >= 92%-->
			<driveSymbol name="OIL_FILL_LEVEL_TO_LOW" node="oilFillLevelToLow"/> <!--active when vehicle damage >= 95%-->
			<driveSymbol name="STOP_IMMINENTLY" node="stopImminently"/> <!--active when vehicle damage = 100%-->
			<driveSymbol name="VEHICLE_SELECTED" node="vehicleSelected"/>
			<driveSymbol name="TOOL_FRONT_SELECTED_1" node="toolFrontSelected1"/>
			<driveSymbol name="TOOL_FRONT_SELECTED_2" node="toolFrontSelected2"/>
			<driveSymbol name="TOOL_BACK_SELECTED_1" node="toolBackSelected1"/>
			<driveSymbol name="TOOL_BACK_SELECTED_2" node="toolBackSelected2"/>
			<driveSymbol name="TOOL_FRONT_UNSELECTED_1" node="toolFrontUnselected1"/>
			<driveSymbol name="TOOL_FRONT_UNSELECTED_2" node="toolFrontUnselected2"/>
			<driveSymbol name="TOOL_BACK_UNSELECTED_1" node="toolBackUnselected1"/>
			<driveSymbol name="TOOL_BACK_UNSELECTED_2" node="toolBackUnselected2"/>
			<driveSymbol name="FRONTLOADER_SELECTED" node="frontloaderIsSelected"/>
			<driveSymbol name="FRONTLOADER_UNSELECTED" node="frontloaderIsUnselected"/>
			<driveSymbol name="FRONTLOADER_TOOL_SELECTED" node="frontloaderToolIsSelected"/>
			<driveSymbol name="FRONTLOADER_TOOL_UNSELECTED" node="frontloaderToolIsUnselected"/>
			<driveSymbol name="TURN_LIGHT_TRAILER_ANY_1" node="iconTurnTrailer1Decal"/>
			<driveSymbol name="TURN_LIGHT_TRAILER_LEFT_1" node="iconTurnTrailer1LeftDecal"/>
			<driveSymbol name="TURN_LIGHT_TRAILER_RIGHT_1" node="iconTurnTrailer1RightDecal"/>
			<driveSymbol name="TURN_LIGHT_TRAILER_ANY_2" node="iconTurnTrailer2Decal"/>
			<driveSymbol name="TURN_LIGHT_TRAILER_LEFT_2" node="iconTurnTrailer2LeftDecal"/>
			<driveSymbol name="TURN_LIGHT_TRAILER_RIGHT_2" node="iconTurnTrailer2RightDecal"/>
			<driveSymbol name="STRAW_CHOPPER" node="strawChopper"/>
            <driveSymbol name="OVERLOADING_ACTIVE" node="overloadingActive"/>
            <driveSymbol name="GRAIN_TANK_UNFOLDED" node="grainTankUnfolded"/>
		</driveSymbols>
		
		<dashboards>
			<dashboard displayType="NUMBER" valueType="fillLevelPercent" numbers="fillLevelPercentNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/>
			<dashboard displayType="NUMBER" valueType="fillLevel" numbers="fillLevelNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/>
			<dashboard displayType="NUMBER" valueType="gpsWorkingWidth" numbers="gpsWorkingWidthNumbers" numberColor="0 0 0 0" precision="1" groups="MOTOR_ACTIVE"/>
			<dashboard displayType="NUMBER" valueType="gpsCurrentLane" numbers="gpsCurrentLaneNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/>
			<dashboard displayType="NUMBER" valueType="currentBaleCount" numbers="currentBaleCountNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/> <!-- needs my 'FS19_BaleCounter.zip'! -->
			<dashboard displayType="NUMBER" valueType="totalBaleCount" numbers="totalBaleCountNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/> <!-- needs my 'FS19_BaleCounter.zip'! -->
			<dashboard displayType="NUMBER" valueType="currentWrappedBaleCount" numbers="currentWrappedBaleCountNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/> <!-- needs my 'FS19_BaleCounter.zip'! -->
			<dashboard displayType="NUMBER" valueType="totalWrappedBaleCount" numbers="totalWrappedBaleCountNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/> <!-- needs my FS19_BaleCounter.zip'! -->
			<dashboard displayType="EMITTER" valueType="lowFuelFillLevel" node="fuelDecal" baseColor="BLACK" emitColor="YELLOW" groups="MOTOR_ACTIVE"/>
			<dashboard displayType="NUMBER" valueType="motorLoad" numbers="motorLoadNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/>
			<dashboard displayType="NUMBER" valueType="wheelSlip" numbers="wheelSlipNumbers" numberColor="0 0 0 0" precision="0" groups="MOTOR_ACTIVE"/>
			<dashboard displayType="NUMBER" valueType="currentAngle" numbers="currentAngleNumbers" numberColor="1 1 1 1" precision="1" groups="MOTOR_ACTIVE"/>
		</dashboards>
	</animatedIndoorParts>
]]

AnimatedIndoorParts = {};
AnimatedIndoorParts.debugPriority = 0;
AnimatedIndoorParts.currentModName = g_currentModName;

function AnimatedIndoorParts.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Drivable, specializations);
end;

function AnimatedIndoorParts.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate",
		"onRegisterActionEvents"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, AnimatedIndoorParts);
	end;
end;

function AnimatedIndoorParts.registerFunctions(vehicleType)
	local newFunctions = {
		"getFillLevelPercentTotal",
		"getFillLevelTotal",
		"getGpsWorkingWidth",
		"getGpsCurrentLane",
		"getCurrentBaleCount",
		"getTotalBaleCount",
		"getCurrentWrappedBaleCount",
		"getTotalWrappedBaleCount",
		"getMotorLoad",
		"getWheelSlip",
		"getFuelUsage",
		"getMotorTemperature",
		"getPsCurrentLane",
		"getPsWorkingWidth",
		"getPsTramlineDistance",
		"getPsMaxLanes",
		"getCurrentFieldNumber",
		"setCurrentScreen",
		"getFillLevelPercentTrailer1",
		"getFillLevelTrailer1",
		"getFillLevelPercentTrailer2",
		"getFillLevelTrailer2"
	};
	
	for _, newFunction in ipairs(newFunctions) do
		SpecializationUtil.registerFunction(vehicleType, newFunction, AnimatedIndoorParts[newFunction]);
	end;
end;

function AnimatedIndoorParts:onLoad(savegame)
	if JohnDeerePackUtil == nil then
		return;
	end;

	local baseKey = "vehicle.animatedIndoorParts";

	AnimatedIndoorParts.debugPriority = JohnDeerePackUtil.getDebugPriority(self.xmlFile, baseKey .. "#debugPriority");
	
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local specMotorized = JohnDeerePackUtil.getSpecByName(self, "motorized");

	specAnimatedIndoorParts.driveSymbols = {};

	local driveSymbolNumber = 0;

	JohnDeerePackUtil.printDebug("", AnimatedIndoorParts.debugPriority, false, "");
	JohnDeerePackUtil.printDebug("-----------------------------------------------Debug from the AnimatedIndoorParts.lua Start------------------------------------------------", AnimatedIndoorParts.debugPriority, false, "");

	while true do
		local driveSymbolKey = baseKey .. ".driveSymbols.driveSymbol(" .. tostring(driveSymbolNumber) .. ")";
		
		if not hasXMLProperty(self.xmlFile, driveSymbolKey) then
			break;
		end;
		
		local driveSymbol = {};
		
		driveSymbol.node = I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, driveSymbolKey .. "#node"), self.i3dMappings);
		
		if driveSymbol.node ~= nil then	
			driveSymbol.name = string.upper(Utils.getNoNil(getXMLString(self.xmlFile, driveSymbolKey .. "#name"), JohnDeerePackUtil.supportetDriveSymbolNames[1]));
			driveSymbol.emitColor = self:getDashboardColor(Utils.getNoNil(getXMLString(self.xmlFile, driveSymbolKey .. "#emitColor"), "GREEN"));
			driveSymbol.baseColor = self:getDashboardColor(Utils.getNoNil(getXMLString(self.xmlFile, driveSymbolKey .. "#baseColor"), "BLACK"));
			driveSymbol.intensity = Utils.getNoNil(getXMLFloat(self.xmlFile, driveSymbolKey .. "#intensity"), 1);
			
			if driveSymbol.name ~= "DISABLE_ON_MOTOR_OFF" then
				if driveSymbol.emitColor == nil then
					JohnDeerePackUtil.printError("Failed to load emit color '" .. getXMLString(self.xmlFile, driveSymbolKey .. "#emitColor") .. "'! (drive symbol number '" .. driveSymbolNumber + 1 .. "')! Use default emit color 'GREEN' instead!", false, false, "AnimatedIndoorParts");
				
					driveSymbol.emitColor = self:getDashboardColor("GREEN");
				end;
			
				if driveSymbol.baseColor == nil then
					JohnDeerePackUtil.printError("Failed to load base color '" .. getXMLString(self.xmlFile, driveSymbolKey .. "#baseColor") .. "'! (drive symbol number '" .. driveSymbolNumber + 1 .. "')! Use default base color 'BLACK' instead!", false, false, "AnimatedIndoorParts");
				
					driveSymbol.baseColor = self:getDashboardColor("BLACK");
				end;
			end;

			if JohnDeerePackUtil.getIsSupportetValue(JohnDeerePackUtil.supportetDriveSymbolNames, driveSymbol.name) then
				setVisibility(driveSymbol.node, false);
				
				if driveSymbol.name == "CONTROL_LAMP" then
					driveSymbol.displayTime = Utils.getNoNil(getXMLFloat(self.xmlFile, driveSymbolKey .. "#displayTime"), 2200);
					driveSymbol.currentTime = 1;
				elseif driveSymbol.name ~= "DISABLE_ON_MOTOR_OFF" then
					setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
					setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
					setShaderParameter(driveSymbol.node, "baseColor", driveSymbol.baseColor[1], driveSymbol.baseColor[2], driveSymbol.baseColor[3], driveSymbol.baseColor[4], false);
				end;

				table.insert(specAnimatedIndoorParts.driveSymbols, driveSymbol);

				JohnDeerePackUtil.printDebug("Load drive symbol '" .. driveSymbol.name .. "' (number '" .. driveSymbolNumber + 1 .. "') successfully.", AnimatedIndoorParts.debugPriority, true, "AnimatedIndoorParts");
			else
				JohnDeerePackUtil.printError("Found unsupportet drive symbol name '" .. driveSymbol.name .. "' (drive symbol number '" .. driveSymbolNumber + 1 .. "')! Supportet names: '" .. JohnDeerePackUtil.getSupportetValues(JohnDeerePackUtil.supportetDriveSymbolNames) .. "'. Skipping this entry!", false, false, "AnimatedIndoorParts");
			end;
		else
			JohnDeerePackUtil.printError("Invalid node for drive symbol number '" .. driveSymbolNumber + 1 .. "'! Skipping this entry!", false, false, "AnimatedIndoorParts");
		end;
		
		driveSymbolNumber = driveSymbolNumber + 1;
	end;

	specAnimatedIndoorParts.fillLevelBars = {};

	local fillLevelBarNumber = 0;
	
	while true do
		local fillLevelBarKey = baseKey .. ".fillLevelBar(" .. tostring(fillLevelBarNumber) .. ")";

		if not hasXMLProperty(self.xmlFile, fillLevelBarKey) then
			break;
		end;

		local fillLevelBar = {};

		local node = I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, fillLevelBarKey .. "#node"), self.i3dMappings);

		if node ~= nil then
			fillLevelBar.node = node;
			fillLevelBar.scaleDirection = Utils.getNoNil(string.upper(getXMLString(self.xmlFile, fillLevelBarKey .. "#scaleDirection")), "X");
			fillLevelBar.isTrailer1 = Utils.getNoNil((getXMLBool(self.xmlFile, fillLevelBarKey .. "#isTrailer1")), false);
			fillLevelBar.isTrailer2 = Utils.getNoNil((getXMLBool(self.xmlFile, fillLevelBarKey .. "#isTrailer2")), false);

			if not JohnDeerePackUtil.getIsSupportetValue(JohnDeerePackUtil.supportetScaleDirections, fillLevelBar.scaleDirection) then 
				JohnDeerePackUtil.printError("Invalid scale direction '" .. fillLevelBar.scaleDirection .. "' for the fill level bar! Supportet scale directions: '" .. JohnDeerePackUtil.getSupportetValues(supportetScaleDirections) .. "'. Using scale direction 'X' instead!", true, false, "AnimatedIndoorParts");

				fillLevelBar.scaleDirection = "X";
			end;

			setScale(fillLevelBar.node, 0, 0, 0);

			table.insert(specAnimatedIndoorParts.fillLevelBars, fillLevelBar);
		end;

		fillLevelBarNumber = fillLevelBarNumber + 1;
	end;

	specAnimatedIndoorParts.screens = {};

	local screenNumber = 0;

	while true do
		local screenKey = baseKey .. ".screens.screen(" .. tostring(screenNumber) .. ")";

		if not hasXMLProperty(self.xmlFile, screenKey) then
			break;
		end;

		local screen = {};

		local node = I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, screenKey .. "#node"), self.i3dMappings);

		if node ~= nil then
			local allowInsertScreen = true;
			
			local configName = Utils.getNoNil(getXMLString(self.xmlFile, screenKey .. "#configName"), "");
			local activeConfigs = Utils.getNoNil(getXMLString(self.xmlFile, screenKey .. "#activeConfigs"), "");
			
    		if configName ~= "" and activeConfig ~= "" then
				allowInsertScreen = false;
				
				local storeItem = g_storeManager:getItemByXMLFilename(self.configFileName);
			
    		    if storeItem ~= nil and storeItem.configurations ~= nil and storeItem.configurations[configName] ~= nil then
					local activeConfigs = StringUtil.splitString(", ", activeConfigs);
    		        local configurations = storeItem.configurations[configName];
    		        local config = configurations[self.configurations[configName]];
				
    		        for _, activeConfig in pairs(activeConfigs) do
						if g_i18n:hasText(activeConfig) then
							activeConfig = g_i18n:getText(activeConfig);
						end;
						
						if config.name == activeConfig then	
							allowInsertScreen = true;

							break;
						end;
    		        end;
    		    end;
    		end;

			if allowInsertScreen then
				screen.node = node;

				setVisibility(screen.node, false);

				table.insert(specAnimatedIndoorParts.screens, screen);
			end;
		end;

		screenNumber = screenNumber + 1
	end;

	if #specAnimatedIndoorParts.screens > 0 then
		local currentScreenNumber = 1;
		
		if savegame ~= nil then
			currentScreenNumber = Utils.getNoNil(getXMLInt(savegame.xmlFile, savegame.key .. "." .. AnimatedIndoorParts.currentModName .. ".animatedIndoorParts#currentScreenNumber"), currentScreenNumber);
		end;

		self:setCurrentScreen(currentScreenNumber);
	end;

	specAnimatedIndoorParts.fillTypeIcons = {};

	local fillTypeIconNumber = 0;

	while true do
		local fillTypeIconKey = baseKey .. ".fillTypeIcon(" .. tostring(fillTypeIconNumber) .. ")";

		if not hasXMLProperty(self.xmlFile, fillTypeIconKey) then
			break;
		end;

		local fillTypeIcon = {};

		local node = I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, fillTypeIconKey .. "#node"), self.i3dMappings);

		if node ~= nil then
			fillTypeIcon.node = node;
			fillTypeIcon.trailerNumber = Utils.getNoNil(getXMLInt(self.xmlFile, fillTypeIconKey .. "#trailerNumber"), 1);

			setVisibility(fillTypeIcon.node, false);

			table.insert(specAnimatedIndoorParts.fillTypeIcons, fillTypeIcon);
		end;

		fillTypeIconNumber = fillTypeIconNumber + 1
	end;

	specAnimatedIndoorParts.fruitTypeIcons = {};

	local fruitTypeIconNumber = 0;

	while true do
		local fruitTypeIconKey = baseKey .. ".fruitTypeIcon(" .. tostring(fruitTypeIconNumber) .. ")";

		if not hasXMLProperty(self.xmlFile, fruitTypeIconKey) then
			break;
		end;

		local fruitTypeIcon = {};

		local node = I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, fruitTypeIconKey .. "#node"), self.i3dMappings);

		if node ~= nil then
			fruitTypeIcon.node = node;
			
			setVisibility(fruitTypeIcon.node, false);

			table.insert(specAnimatedIndoorParts.fruitTypeIcons, fruitTypeIcon);
		end;

		fruitTypeIconNumber = fruitTypeIconNumber + 1
	end;

	JohnDeerePackUtil.printDebug("", AnimatedIndoorParts.debugPriority, false, "");
	JohnDeerePackUtil.printDebug("-----------------------------------------------Debug from the AnimatedIndoorParts.lua End------------------------------------------------", AnimatedIndoorParts.debugPriority, false, "");

	specAnimatedIndoorParts.gpsModName = "";
	specAnimatedIndoorParts.lsaModName = "";
	specAnimatedIndoorParts.bsaModName = "";
	specAnimatedIndoorParts.psModName = "";
	
	for _, mod in pairs(g_modManager.mods) do
		if _G[tostring(mod.modName)].GlobalPositioningSystem ~= nil then		
			if g_modIsLoaded[tostring(mod.modName)] then	
				specAnimatedIndoorParts.gpsModName = mod.modName;

				JohnDeerePackUtil.printDebug("Found GPS Mod with name '" .. mod.modName .. ".zip'.", AnimatedIndoorParts.debugPriority, true, "AnimatedIndoorParts");

				break;
			end;
		end;
	end;

	for _, mod in pairs(g_modManager.mods) do
		if mod.title == "Lenkachse sperren" or mod.title == "Lock steering axle" then	
			if g_modIsLoaded[tostring(mod.modName)] then	
				specAnimatedIndoorParts.lsaModName = mod.modName;

				JohnDeerePackUtil.printDebug("Found Lock Steering Axles Mod with name '" .. mod.modName .. ".zip'.", AnimatedIndoorParts.debugPriority, true, "AnimatedIndoorParts");

				break;
			end;
		end;
	end;

	for _, mod in pairs(g_modManager.mods) do
		if _G[tostring(mod.modName)].BuyableSteeringAxle ~= nil then		
			if g_modIsLoaded[tostring(mod.modName)] then	
				specAnimatedIndoorParts.bsaModName = mod.modName;

				JohnDeerePackUtil.printDebug("Found Buyable Steering Axle in '" .. mod.modName .. ".zip'.", AnimatedIndoorParts.debugPriority, true, "AnimatedIndoorParts");

				break;
			end;
		end;
	end;

	for _, mod in pairs(g_modManager.mods) do
		if _G[tostring(mod.modName)].ProSeedTramLines ~= nil then		
			if g_modIsLoaded[tostring(mod.modName)] then	
				specAnimatedIndoorParts.psModName = mod.modName;

				JohnDeerePackUtil.printDebug("Found Pro Seed Mod with name '" .. mod.modName .. ".zip'.", AnimatedIndoorParts.debugPriority, true, "AnimatedIndoorParts");

				break;
			end;
		end;
	end;
	
	specAnimatedIndoorParts.isLowFillLevel = false;
	specAnimatedIndoorParts.cardinalDirection = 0;
	specAnimatedIndoorParts.dt = 0;
	specAnimatedIndoorParts.currentFieldNumber = 0;
	
	if self.loadDashboardsFromXML ~= nil then
		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "fillLevelPercentTotal",
				valueObject = self,
				valueFunc = "getFillLevelPercentTotal",
				minFunc = 0,
				maxFunc = 100
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "fillLevelTotal",
				valueObject = self,
				valueFunc = "getFillLevelTotal",
				minFunc = 0,
				maxFunc = 999999
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "fillLevelPercentTrailer1",
				valueObject = self,
				valueFunc = "getFillLevelPercentTrailer1",
				minFunc = 0,
				maxFunc = 100
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "fillLevelTrailer1",
				valueObject = self,
				valueFunc = "getFillLevelTrailer1",
				minFunc = 0,
				maxFunc = 999999
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "fillLevelPercentTrailer2",
				valueObject = self,
				valueFunc = "getFillLevelPercentTrailer2",
				minFunc = 0,
				maxFunc = 100
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "fillLevelTrailer2",
				valueObject = self,
				valueFunc = "getFillLevelTrailer2",
				minFunc = 0,
				maxFunc = 999999
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "lowFuelFillLevel",
				valueObject = specAnimatedIndoorParts,
				valueFunc = "isLowFillLevel"
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards",

			{
				valueTypeToLoad = "currentBaleCount",
				valueObject = self,
				valueFunc = "getCurrentBaleCount",
				minFunc = 0,
				maxFunc = 999999
			}
		);
		
		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards",

			{
				valueTypeToLoad = "totalBaleCount",
				valueObject = self,
				valueFunc = "getTotalBaleCount",
				minFunc = 0,
				maxFunc = 999999
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards",

			{
				valueTypeToLoad = "currentWrappedBaleCount",
				valueObject = self,
				valueFunc = "getCurrentWrappedBaleCount",
				minFunc = 0,
				maxFunc = 999999
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards",

			{
				valueTypeToLoad = "totalWrappedBaleCount",
				valueObject = self,
				valueFunc = "getTotalWrappedBaleCount",
				minFunc = 0,
				maxFunc = 999999
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards",

			{
				valueTypeToLoad = "motorLoad",
				valueObject = self,
				valueFunc = "getMotorLoad",
				minFunc = 0,
				maxFunc = 100
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards",

			{
				valueTypeToLoad = "wheelSlip",
				valueObject = self,
				valueFunc = "getWheelSlip",
				minFunc = 0,
				maxFunc = 100
			}
		);

		self:loadDashboardsFromXML(
			self.xmlFile, 
			"vehicle.animatedIndoorParts.dashboards", 
			
			{
				valueTypeToLoad = "currentAngle",
				valueObject = specAnimatedIndoorParts,
				valueFunc = "cardinalDirection",
				minFunc = 0,
				maxFunc = 360
			}
		);

		if specAnimatedIndoorParts.gpsModName ~= "" then
			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "gpsWorkingWidth",
					valueObject = self,
					valueFunc = "getGpsWorkingWidth",
					minFunc = 0,
					maxFunc = 99
				}
			);

			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "gpsCurrentLane",
					valueObject = self,
					valueFunc = "getGpsCurrentLane",
					minFunc = 0,
					maxFunc = 9999
				}
			);

			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "currentFieldNumber",
					valueObject = self,
					valueFunc = "getCurrentFieldNumber",
					minFunc = 0,
					maxFunc = 9999
				}
			);
		end;

		if specMotorized ~= nil then
			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "currentFuelUsage",
					valueObject = self,
					valueFunc = "getFuelUsage",
					minFunc = 0,
					maxFunc = 999
				}
			);
			
			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "currentMotorTemperature",
					valueObject = self,
					valueFunc = "getMotorTemperature",
					minFunc = 0,
					maxFunc = 999999
				}
			);
		end;

		if specAnimatedIndoorParts.psModName ~= "" then
			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "psCurrentLane",
					valueObject = self,
					valueFunc = "getPsCurrentLane",
					minFunc = 0,
					maxFunc = 12
				}
			);

			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "psMaxLanes",
					valueObject = self,
					valueFunc = "getPsMaxLanes",
					minFunc = 0,
					maxFunc = 12
				}
			);

			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "psWorkingWidth",
					valueObject = self,
					valueFunc = "getPsWorkingWidth",
					minFunc = 0,
					maxFunc = 99
				}
			);

			self:loadDashboardsFromXML(
				self.xmlFile, 
				"vehicle.animatedIndoorParts.dashboards", 

				{
					valueTypeToLoad = "psTramlineDistance",
					valueObject = self,
					valueFunc = "getPsTramlineDistance",
					minFunc = 0,
					maxFunc = 72
				}
			);
		end;
	end;

	if self.spec_enterable.cameras ~= nil then
		for cameraNumber, camera in pairs(self.spec_enterable.cameras) do
			if camera.isInside then
				camera.oldFovSaved = math.deg(getFovY(camera.cameraNode));
			end;
		end;
	end;

	specAnimatedIndoorParts.inputIsPressed = false;
	specAnimatedIndoorParts.resetIndoorCamera = false;
end;

function AnimatedIndoorParts:saveToXMLFile(xmlFile, key, usedModNames)
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");

	if #specAnimatedIndoorParts.screens > 0 and specAnimatedIndoorParts.currentScreenNumber > 1 then
		setXMLInt(xmlFile, key .. "#currentScreenNumber", specAnimatedIndoorParts.currentScreenNumber);
	end;
end;

function AnimatedIndoorParts:onRegisterActionEvents(isActiveForInput)
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	
	if self.isClient then
		self:clearActionEventsTable(specAnimatedIndoorParts.actionEvents);
        
		if self:getIsActiveForInput(true) then
            local actionEventId;
			
			for _, input in pairs({"INDOOR_CAMERA_ZOOM_BUTTON", "INDOOR_CAMERA_RESET_BUTTON"}) do
				_, actionEventId = self:addActionEvent(specAnimatedIndoorParts.actionEvents, InputAction[input], self, AnimatedIndoorParts.setIndoorCamera, false, true, true, true, nil);

				g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL);
				g_inputBinding:setActionEventTextVisibility(actionEventId, true);
				g_inputBinding:setActionEventActive(actionEventId, true);
				g_inputBinding:setActionEventText(actionEventId, g_i18n:getText("input_" .. input));
			end;

			_, actionEventId = self:addActionEvent(specAnimatedIndoorParts.actionEvents, InputAction.SWITCH_SCREEN_BUTTON, self, AnimatedIndoorParts.switchScreen, false, true, false, true, nil);

			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL);
			g_inputBinding:setActionEventTextVisibility(actionEventId, #specAnimatedIndoorParts.screens > 0);
			g_inputBinding:setActionEventActive(actionEventId, #specAnimatedIndoorParts.screens > 0);
			g_inputBinding:setActionEventText(actionEventId, g_i18n:getText("input_SWITCH_SCREEN_BUTTON"));
		end;
	end;
end;

function AnimatedIndoorParts.switchScreen(self, actionName, inputValue, callbackState, isAnalog)
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");

	if #specAnimatedIndoorParts.screens > 0 then
		if specAnimatedIndoorParts.currentScreenNumber < #specAnimatedIndoorParts.screens then
			self:setCurrentScreen(specAnimatedIndoorParts.currentScreenNumber + 1);
		else
			self:setCurrentScreen(1);
		end;
	end;
end;

function AnimatedIndoorParts:setCurrentScreen(currentScreenNumber, noEventSend)
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");

    if currentScreenNumber ~= specAnimatedIndoorParts.currentScreenNumber then
		if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(AnimatedIndoorPartsScreenEvent:new(self, currentScreenNumber), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(AnimatedIndoorPartsScreenEvent:new(self, currentScreenNumber));
			end;
		end;

		specAnimatedIndoorParts.currentScreenNumber = currentScreenNumber;
	end;
end;

function AnimatedIndoorParts.setIndoorCamera(self, actionName, inputValue, callbackState, isAnalog)
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");

	if actionName == "INDOOR_CAMERA_ZOOM_BUTTON" then
		specAnimatedIndoorParts.inputIsPressed = true;
	else
		specAnimatedIndoorParts.resetIndoorCamera = true;
	end;
end;

function AnimatedIndoorParts:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local specAttachJoints = JohnDeerePackUtil.getSpecByName(self, "attacherJoints");
	local specFillUnit = JohnDeerePackUtil.getSpecByName(self, "fillUnit");
	local specDrivable = JohnDeerePackUtil.getSpecByName(self, "drivable");
	local specWearable = JohnDeerePackUtil.getSpecByName(self, "wearable");
	local specTrailer = JohnDeerePackUtil.getSpecByName(self, "trailer");
	local specMotorized = JohnDeerePackUtil.getSpecByName(self, "motorized");
	local specCombine = JohnDeerePackUtil.getSpecByName(self, "combine");
	local specDischargeable = JohnDeerePackUtil.getSpecByName(self, "dischargeable");
	local specPipe = JohnDeerePackUtil.getSpecByName(self, "pipe");

	local isTruckABSActive, truckIsTipping, airCompressorDoRefill = false, false, false;
	local warningWorkshop1, warningWorkshop2, brakesWearedOut, stopImminently, motorNeedMaintrace, oilFillLevelToLow, lowOilPressure = false, false, false, false, false, false;
	local vehicleSchemaIsTurnedOn = {false, false, false, false};
	local isStrawChopperActive, isOverloading, isGrainTankUnfolded, isPipeUnfolded = false, false, false, false;
	
	local dirX, _, dirZ = localDirectionToWorld(self.rootNode, 0, 0, 1);
	
	specAnimatedIndoorParts.cardinalDirection = MathUtil.round(math.deg(math.abs(MathUtil.getYRotationFromDirection(dirX, dirZ) - math.pi)), 1);
	specAnimatedIndoorParts.dt = dt;

	if specTrailer ~= nil and specTrailer.tipState ~= nil then
		truckIsTipping = specTrailer.tipState ~= Trailer.TIPSTATE_CLOSED;
	end;

	if self.getIsTurnedOn ~= nil and self:getIsTurnedOn() then
		vehicleSchemaIsTurnedOn[1] = true;
	end;
	
	if specCombine ~= nil then
		isStrawChopperActive = not specCombine.isSwathActive;
		isGrainTankUnfolded = self:getToggledFoldDirection() == 1;
	end;

	if specDischargeable ~= nil then
		isOverloading = specDischargeable.currentDischargeState ~= Dischargeable.DISCHARGE_STATE_OFF;
	end;

	if specPipe ~= nil then
		isPipeUnfolded = specPipe.targetState == specPipe.numStates;
	end;

	if specDrivable ~= nil then	
		isTruckABSActive = specDrivable.axisForward <= -1 and self.movingDirection > 0 and self:getLastSpeed() > 10;
	end;

	for _, fillUnit in pairs(specFillUnit.fillUnits) do
		if fillUnit.fillType == FillType.DIESEL then
			specAnimatedIndoorParts.isLowFillLevel = fillUnit.fillLevel / fillUnit.capacity < 0.2;
		end;
	end;

	if specWearable ~= nil then
		warningWorkshop1 = self:getWearTotalAmount() >= 0.7;
		warningWorkshop2 = self:getWearTotalAmount() >= 0.8;
		brakesWearedOut = self:getWearTotalAmount() >= 0.82;
		motorNeedMaintrace = self:getWearTotalAmount() >= 0.9;
		lowOilPressure = self:getWearTotalAmount() >= 0.92;
		oilFillLevelToLow = self:getWearTotalAmount() >= 0.95;
		stopImminently = self:getWearTotalAmount() >= 1;
	end;

	if specMotorized ~= nil and specMotorized.consumersByFillTypeName ~= nil and specMotorized.consumersByFillTypeName.air ~= nil then
		airCompressorDoRefill = Utils.getNoNil(specMotorized.consumersByFillTypeName.air.doRefill, false);
	end;

	if #specAnimatedIndoorParts.fillLevelBars > 0 then
		for _, fillLevelBar in pairs(specAnimatedIndoorParts.fillLevelBars) do
			setVisibility(fillLevelBar.node, self:getIsMotorStarted());

			local percentage = self:getFillLevelPercentTotal() / 100;
			local percentageTrailer1 = self:getFillLevelPercentTrailer1() / 100;
			local percentageTrailer2 = self:getFillLevelPercentTrailer2() / 100;

			if fillLevelBar.isTrailer1 then
				percentage = percentageTrailer1;
			elseif fillLevelBar.isTrailer2 then
				percentage = percentageTrailer2;
			end;

			if fillLevelBar.scaleDirection == "X" then
				setScale(fillLevelBar.node, percentage * 1, 1, 1);
			elseif fillLevelBar.scaleDirection == "Y" then
				setScale(fillLevelBar.node, 1, percentage * 1, 1);
			elseif fillLevelBar.scaleDirection == "Z" then
				setScale(fillLevelBar.node, 1, 1, percentage * 1);
			elseif fillLevelBar.scaleDirection == "ALL" then
				setScale(fillLevelBar.node, percentage * 1, percentage * 1, percentage * 1);
			end;
		end;
	end;

	if #specAnimatedIndoorParts.screens > 0 then
		for screenNumber, screen in pairs(specAnimatedIndoorParts.screens) do
			setVisibility(screen.node, screenNumber == specAnimatedIndoorParts.currentScreenNumber and self:getIsMotorStarted());
		end;
	end;

	if specAttachJoints ~= nil then
		local isPTOFrontActive, isPTOBackActive, moveDownFront, moveDownBack = false, false, false, false;
		local isLeftTipping1, isRightTipping1, isBackTipping1, isGrainDoorTipping1 = false, false, false, false;
		local isLeftTipping2, isRightTipping2, isBackTipping2, isGrainDoorTipping2 = false, false, false, false;
		local numberOfAttachedTrailers, numberOfAttachedTrailersWithTurnLights, trailerIsTipping, trailerIsTipping1, trailerIsTipping2, trailerHasSteerAxle = 0, 0, false, false, false, false;
		local diffLockFront, diffLockBack, handbrakeIsActive, fourWheel = false, false, false, false;
		local gpsIsActive, gpsSteeringIsActive, gpsCurrentLane = false, false, 0;
		local steerAxleIsLocked, ridgeMarkerState, isPickupLowered, toolFrontIsUnfolded, toolBackIsUnfolded, hasFoldablePartsBack, hasFoldablePartsFront = false, 0, false, false, false, false, false;
		local cutterSpeed, currentFillType1, currentFillType2, currentFruitType = 0, FillType.UNKNOWN, FillType.UNKNOWN, FillType.UNKNOWN;
		local drivableIsSelected, toolFrontIsSelected1, toolBackIsSelected1, toolFrontIsAttached1, toolBackIsAttached1 = self:getIsSelected(), false, false, false, false;
		local toolFrontIsSelected2, toolBackIsSelected2, toolFrontIsAttached2, toolBackIsAttached2 = false, false, false, false;
		local frontloaderIsSelected, frontloaderToolIsSelected, frontloaderIsAttached, frontloaderToolIsAttached = false, false, false, false;
		local currentShutoffModeIsLeft, currentShutoffModeIsRight, currentModeIsAuto, currentModeIsSemi, currentModeIsManual, createTramLines = false, false, false, false, false, false;
		
		if specAnimatedIndoorParts.gpsModName ~= "" then
			local specGlobalPositioningSystem = JohnDeerePackUtil.getSpecByName(self, "globalPositioningSystem", specAnimatedIndoorParts.gpsModName);
			
			if specGlobalPositioningSystem.guidanceData ~= nil then
				gpsIsActive = specGlobalPositioningSystem.guidanceIsActive;
				gpsSteeringIsActive = specGlobalPositioningSystem.guidanceSteeringIsActive;
				gpsCurrentLane = specGlobalPositioningSystem.guidanceData.currentLane;
			end;
		end;
	
		if self.vcaDiffLockFront == nil then
			if self.vData ~= nil then
				diffLockFront, diffLockBack, handbrakeIsActive, fourWheel = self.vData.is[1], self.vData.is[2], self.vData.is[5] and self.vData.is[6], self.vData.want[3] == 1;
			end;
		else
			diffLockFront, diffLockBack, handbrakeIsActive, fourWheel = self.vcaDiffLockFront and self.vcaDiffLockAWD, self.vcaDiffLockBack, self.vcaNewHandbrake, self.vcaDiffLockAWD;
		end;

		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local object_specAttachable = JohnDeerePackUtil.getSpecByName(object, "attachable");
			local object_specTrailer = JohnDeerePackUtil.getSpecByName(object, "trailer");
			local object_specLockSteeringAxles = JohnDeerePackUtil.getSpecByName(object, "lockSteeringAxles", specAnimatedIndoorParts.lsaModName);
			local object_specBuyableSteeringAxle = JohnDeerePackUtil.getSpecByName(object, "buyableSteeringAxle", specAnimatedIndoorParts.bsaModName);
			local object_specRidgeMarker = JohnDeerePackUtil.getSpecByName(object, "ridgeMarker");
			local object_specFoldable = JohnDeerePackUtil.getSpecByName(object, "foldable");
			local object_specCutter = JohnDeerePackUtil.getSpecByName(object, "cutter");
			local object_specLights = JohnDeerePackUtil.getSpecByName(object, "lights");
			local object_specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");
			local object_specSowingMachine = JohnDeerePackUtil.getSpecByName(object, "sowingMachine");
			local attacherVehicle_zScale = nil;

			if specAnimatedIndoorParts.psModName ~= "" then
				local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);

				if specProSeedTramLines ~= nil then
					currentModeIsAuto = specProSeedTramLines.tramLineMode == 2;
					currentModeIsSemi = specProSeedTramLines.tramLineMode == 1;
					currentModeIsManual = specProSeedTramLines.tramLineMode == 0;
					currentShutoffModeIsLeft = specProSeedTramLines.shutoffMode == 1;
					currentShutoffModeIsRight = specProSeedTramLines.shutoffMode == 2;
					createTramLines = specProSeedTramLines.createTramLines;
				end;
			end;
			
			if object_specCutter ~= nil then
				for animationNumber, animationNode in pairs(object_specCutter.animationNodes) do
					if getName(animationNode.node) == "reelRotSpeed" then
						local maxSpeed = math.deg(animationNode.rotSpeed) * 1000;
						local divident = 1;
						local multiplikator = 2;
						
						if string.find(maxSpeed, "-") then
							maxSpeed = -maxSpeed;
							divident = 2;
							multiplikator = 1;
						end;

						if object_specCutter.cutterSpeed == nil then
							object_specCutter.cutterSpeed = 0;
						end;

						if self:getIsTurnedOn() then
							if object_specCutter.cutterSpeed < maxSpeed then	
								object_specCutter.cutterSpeed = math.min(object_specCutter.cutterSpeed + (animationNode.turnOnFadeTime / 1000) * multiplikator, maxSpeed);
							end;

							vehicleSchemaIsTurnedOn[2] = true;
						else
							if object_specCutter.cutterSpeed > 0 then	
								object_specCutter.cutterSpeed = math.max(object_specCutter.cutterSpeed - (animationNode.turnOffFadeTime / 1000) / divident, 0);
							end;
						end;

						cutterSpeed = object_specCutter.cutterSpeed;

						JohnDeerePackUtil.renderText(0.5, 0.5, 0.02, "cutterSpeed = " .. cutterSpeed, AnimatedIndoorParts.debugPriority);
					end;
				end;
			end;
			
			if object_specRidgeMarker ~= nil then
				ridgeMarkerState = object_specRidgeMarker.ridgeMarkerState;
			end;

			if object_specLockSteeringAxles ~= nil and object_specLockSteeringAxles.lockSteeringAxle ~= nil then
				steerAxleIsLocked = object_specLockSteeringAxles.lockSteeringAxle;
				trailerHasSteerAxle = true;
			end;

			if object_specBuyableSteeringAxle ~= nil and object_specBuyableSteeringAxle.lockSteeringAxle ~= nil then
				steerAxleIsLocked = object_specBuyableSteeringAxle.lockSteeringAxle;
				trailerHasSteerAxle = true;
			end;
			
			if object_specTrailer ~= nil then
				trailerIsTipping = object_specTrailer.tipState ~= nil and object_specTrailer.tipState ~= Trailer.TIPSTATE_CLOSED;
				trailerIsTipping1 = trailerIsTipping;
				
				if object_specTrailer.preferedTipSideIndex ~= nil then
					local tipSide = object_specTrailer.tipSides[object_specTrailer.preferedTipSideIndex];
					
					if tipSide ~= nil then
						isLeftTipping1 = tipSide.name == g_i18n:getText("info_tipSideLeft");
						isRightTipping1 = tipSide.name == g_i18n:getText("info_tipSideRight");
						isBackTipping1 = tipSide.name == g_i18n:getText("info_tipSideBack");
						isGrainDoorTipping1 = tipSide.name == g_i18n:getText("info_tipSideBackGrainDoor");
					end;
				end;

				numberOfAttachedTrailers = numberOfAttachedTrailers + 1;
			end;

			if object_specLights ~= nil then
				if object_specLights.hasTurnLights or #object_specLights.shaderLeftTurnLights > 0 and #object_specLights.shaderRightTurnLights > 0 then 
					numberOfAttachedTrailersWithTurnLights = numberOfAttachedTrailersWithTurnLights + 1;
				end;
			end;

			if object_specFillUnit ~= nil then
				currentFillType1 = object:getFillUnitFillType(1);
			end;

			if object_specSowingMachine ~= nil then
				currentFruitType = g_fruitTypeManager:getFillTypeIndexByFruitTypeIndex(object_specSowingMachine.workAreaParameters.seedsFruitType);
			end;
			
			if object_specAttachable.attacherVehicle ~= nil then
				local attacherJointVehicleSpec = JohnDeerePackUtil.getSpecByName(object_specAttachable.attacherVehicle, "attacherJoints");
				local implementIndex = object_specAttachable.attacherVehicle:getImplementIndexByObject(object);
				local implement = attacherJointVehicleSpec.attachedImplements[implementIndex];
				local inputJointDescIndex = object_specAttachable.inputAttacherJointDescIndex;
				local jointDescIndex = implement.jointDescIndex;
				local jointDesc = attacherJointVehicleSpec.attacherJoints[jointDescIndex];
				
				if jointDesc.bottomArm ~= nil then
					if jointDesc.bottomArm.zScale == 1 then
						moveDownFront = object:getIsImplementChainLowered();
						
						if object.getIsTurnedOn ~= nil then	
							isPTOFrontActive = object:getIsTurnedOn();
							vehicleSchemaIsTurnedOn[2] = object:getIsTurnedOn();
						end;

						if object_specFoldable ~= nil then
							hasFoldablePartsFront = object_specFoldable.foldingParts ~= nil and #object_specFoldable.foldingParts > 0;
							
							if object_specFoldable.startAnimTime == 1 and object_specFoldable.turnOnFoldMinLimit == 0 then
								toolFrontIsUnfolded = object:getToggledFoldDirection() == 1 and hasFoldablePartsFront;
							elseif object_specFoldable.startAnimTime == 0 or object_specFoldable.turnOnFoldMinLimit > 0 then
								toolFrontIsUnfolded = object:getToggledFoldDirection() == -1 and hasFoldablePartsFront;
							end;
						end;

						toolFrontIsSelected1 = object:getIsSelected();
						toolFrontIsAttached1 = true;

					elseif jointDesc.bottomArm.zScale == -1 then
						moveDownBack = object:getIsImplementChainLowered();
						
						if object.getIsTurnedOn ~= nil then
							isPTOBackActive = object:getIsTurnedOn();
							vehicleSchemaIsTurnedOn[3] = object:getIsTurnedOn();
						end;

						if object_specFoldable ~= nil then
							hasFoldablePartsBack = object_specFoldable.foldingParts ~= nil and #object_specFoldable.foldingParts > 0;
							
							if object_specFoldable.startAnimTime == 1 and object_specFoldable.turnOnFoldMinLimit == 0 then
								toolBackIsUnfolded = object:getToggledFoldDirection() == 1 and hasFoldablePartsBack;
							elseif object_specFoldable.startAnimTime == 0 or object_specFoldable.turnOnFoldMinLimit > 0 then
								toolBackIsUnfolded = object:getToggledFoldDirection() == -1 and hasFoldablePartsBack;
							end;
						end;

						toolBackIsSelected1 = object:getIsSelected();
						toolBackIsAttached1 = true;
					end;

					attacherVehicle_zScale = jointDesc.bottomArm.zScale;
				else
					if object.getIsTurnedOn ~= nil then	
						if not jointDesc.invertX then
							isPTOBackActive = object:getIsTurnedOn();
							vehicleSchemaIsTurnedOn[3] = object:getIsTurnedOn();
							
							if not moveDownBack then
								moveDownBack = object:getIsImplementChainLowered();
							end;

							if jointDesc.jointType ~= AttacherJoints.JOINTTYPE_FRONTLOADER then
								toolBackIsSelected1 = object:getIsSelected();
								toolBackIsAttached1 = true;
							end;
						else
							isPTOFrontActive = object:getIsTurnedOn();
							vehicleSchemaIsTurnedOn[2] = object:getIsTurnedOn();

							if jointDesc.jointType ~= AttacherJoints.JOINTTYPE_FRONTLOADER then
								toolFrontIsSelected1 = object:getIsSelected();
								toolFrontIsAttached1 = true;
							end;
						end;
					else
						if jointDesc.jointType ~= AttacherJoints.JOINTTYPE_ATTACHABLEFRONTLOADER 
							and jointDesc.jointType ~= AttacherJoints.JOINTTYPE_ATTACHABLEFRONTLOADERHAUER 
							and jointDesc.jointType ~= AttacherJoints.JOINTTYPE_FRONTLOADER 
						then
							if not jointDesc.invertX then
								toolBackIsSelected1 = object:getIsSelected();
								toolBackIsAttached1 = true;
							else
								toolFrontIsSelected1 = object:getIsSelected();
								toolFrontIsAttached1 = true;
							end;
						else
							if jointDesc.jointType ~= AttacherJoints.JOINTTYPE_FRONTLOADER then
								frontloaderIsSelected = object:getIsSelected();
								frontloaderIsAttached = true;
							end;
						end;
					end;
				end;
			end;

			local object_specAttachJoints = JohnDeerePackUtil.getSpecByName(object, "attacherJoints");

			if object_specAttachJoints ~= nil then
				for attachedImplement = 1, #object_specAttachJoints.attachedImplements do
					local object = object_specAttachJoints.attachedImplements[attachedImplement].object;
					local object_specAttachable = JohnDeerePackUtil.getSpecByName(object, "attachable");
					local object_specTrailer = JohnDeerePackUtil.getSpecByName(object, "trailer");
					local object_specLights = JohnDeerePackUtil.getSpecByName(object, "lights");
					local object_specRidgeMarker = JohnDeerePackUtil.getSpecByName(object, "ridgeMarker");
					local object_specFoldable = JohnDeerePackUtil.getSpecByName(object, "foldable");
					local object_specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");
					local object_specSowingMachine = JohnDeerePackUtil.getSpecByName(object, "sowingMachine");
					
					local attacherJointVehicleSpec = JohnDeerePackUtil.getSpecByName(object_specAttachable.attacherVehicle, "attacherJoints");
					local implementIndex = object_specAttachable.attacherVehicle:getImplementIndexByObject(object);
					local implement = attacherJointVehicleSpec.attachedImplements[implementIndex];
					local inputJointDescIndex = object_specAttachable.inputAttacherJointDescIndex;
					local jointDescIndex = implement.jointDescIndex;
					local jointDesc = attacherJointVehicleSpec.attacherJoints[jointDescIndex];

					if specAnimatedIndoorParts.psModName ~= "" then
						local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
		
						if specProSeedTramLines ~= nil then
							currentModeIsAuto = specProSeedTramLines.tramLineMode == 2;
							currentModeIsSemi = specProSeedTramLines.tramLineMode == 1;
							currentModeIsManual = specProSeedTramLines.tramLineMode == 0;
							currentShutoffModeIsLeft = specProSeedTramLines.shutoffMode == 1;
							currentShutoffModeIsRight = specProSeedTramLines.shutoffMode == 2;
							createTramLines = specProSeedTramLines.createTramLines;
						end;
					end;

					if jointDesc.jointType ~= AttacherJoints.JOINTTYPE_FRONTLOADER then
						if attacherVehicle_zScale ~= nil then	
							if attacherVehicle_zScale == 1 then
								toolFrontIsSelected2 = object:getIsSelected();
								toolFrontIsAttached2 = true;
							elseif attacherVehicle_zScale == -1 then
								toolBackIsSelected2 = object:getIsSelected();
								toolBackIsAttached2 = true;
								moveDownBack = object:getIsImplementChainLowered();
								
								if object.getIsTurnedOn ~= nil then
									if not isPTOBackActive then	
										isPTOBackActive = object:getIsTurnedOn();
									end;

									vehicleSchemaIsTurnedOn[4] = object:getIsTurnedOn();
								end;

								if object_specFoldable ~= nil then
									hasFoldablePartsBack = object_specFoldable.foldingParts ~= nil and #object_specFoldable.foldingParts > 0;
									
									if object_specFoldable.startAnimTime == 1 and object_specFoldable.turnOnFoldMinLimit == 0 then
										toolBackIsUnfolded = object:getToggledFoldDirection() == 1 and hasFoldablePartsBack;
									elseif object_specFoldable.startAnimTime == 0 or object_specFoldable.turnOnFoldMinLimit > 0 then
										toolBackIsUnfolded = object:getToggledFoldDirection() == -1 and hasFoldablePartsBack;
									end;
								end;
							end;
						else
							if not jointDesc.invertX then
								toolBackIsSelected2 = object:getIsSelected();
								toolBackIsAttached2 = true;
								moveDownBack = object:getIsImplementChainLowered();
								
								if object.getIsTurnedOn ~= nil then
									if not isPTOBackActive then	
										isPTOBackActive = object:getIsTurnedOn();
									end;

									vehicleSchemaIsTurnedOn[4] = object:getIsTurnedOn();
								end;

								if object_specFoldable ~= nil then
									hasFoldablePartsBack = object_specFoldable.foldingParts ~= nil and #object_specFoldable.foldingParts > 0;
									
									if object_specFoldable.startAnimTime == 1 and object_specFoldable.turnOnFoldMinLimit == 0 then
										toolBackIsUnfolded = object:getToggledFoldDirection() == 1 and hasFoldablePartsBack;
									elseif object_specFoldable.startAnimTime == 0 or object_specFoldable.turnOnFoldMinLimit > 0 then
										toolBackIsUnfolded = object:getToggledFoldDirection() == -1 and hasFoldablePartsBack;
									end;
								end;
							else
								toolFrontIsSelected2 = object:getIsSelected();
								toolFrontIsAttached2 = true;
							end;
						end;
					else
						frontloaderToolIsSelected = object:getIsSelected();
						frontloaderToolIsAttached = true;
					end;

					if object_specRidgeMarker ~= nil then
						ridgeMarkerState = object_specRidgeMarker.ridgeMarkerState;
					end;
				
					if object_specTrailer ~= nil then
						trailerIsTipping = object_specTrailer.tipState ~= nil and object_specTrailer.tipState ~= Trailer.TIPSTATE_CLOSED;
						trailerIsTipping2 = trailerIsTipping;

						if object_specTrailer.preferedTipSideIndex ~= nil then
							local tipSide = object_specTrailer.tipSides[object_specTrailer.preferedTipSideIndex];
							
							if tipSide ~= nil then
								isLeftTipping2 = tipSide.name == g_i18n:getText("info_tipSideLeft");
								isRightTipping2 = tipSide.name == g_i18n:getText("info_tipSideRight");
								isBackTipping2 = tipSide.name == g_i18n:getText("info_tipSideBack");
								isGrainDoorTipping2 = tipSide.name == g_i18n:getText("info_tipSideBackGrainDoor");
							end;
						end;

						numberOfAttachedTrailers = numberOfAttachedTrailers + 1;
					end;

					if object_specLights ~= nil then
						if object_specLights.hasTurnLights or #object_specLights.shaderLeftTurnLights > 0 and #object_specLights.shaderRightTurnLights > 0 then 
							numberOfAttachedTrailersWithTurnLights = numberOfAttachedTrailersWithTurnLights + 1;
						end;
					end;

					if object_specFillUnit ~= nil then
						currentFillType2 = object:getFillUnitFillType(1);
					end;

					if object_specSowingMachine ~= nil then
						currentFruitType = g_fruitTypeManager:getFillTypeIndexByFruitTypeIndex(object_specSowingMachine.workAreaParameters.seedsFruitType);
					end;
				end;
			end;
		end;

		if #specAnimatedIndoorParts.fillTypeIcons > 0 then
			for _, fillTypeIcon in pairs(specAnimatedIndoorParts.fillTypeIcons) do
				if fillTypeIcon.node ~= nil then
					if fillTypeIcon.trailerNumber == 1 then
						setVisibility(fillTypeIcon.node, currentFillType1 ~= FillType.UNKNOWN);

						if currentFillType1 ~= FillType.UNKNOWN then
							local oldMaterial = getMaterial(fillTypeIcon.node, 0);
							local newMaterial = g_materialManager:getMaterial(currentFillType1, "icon", 1);

							setVisibility(fillTypeIcon.node, newMaterial ~= nil);
						
							if newMaterial ~= nil then
								ConfigurationUtil.replaceMaterialRec(self, fillTypeIcon.node, oldMaterial, newMaterial);
							end;
						end;
					elseif fillTypeIcon.trailerNumber == 2 then
						setVisibility(fillTypeIcon.node, currentFillType2 ~= FillType.UNKNOWN);

						if currentFillType2 ~= FillType.UNKNOWN then
							local oldMaterial = getMaterial(fillTypeIcon.node, 0);
							local newMaterial = g_materialManager:getMaterial(currentFillType2, "icon", 1);

							setVisibility(fillTypeIcon.node, newMaterial ~= nil);

							if newMaterial ~= nil then
								ConfigurationUtil.replaceMaterialRec(self, fillTypeIcon.node, oldMaterial, newMaterial);
							end;
						end;
					end;
				end;
			end;
		end;

		if #specAnimatedIndoorParts.fruitTypeIcons > 0 then
			for _, fruitTypeIcon in pairs(specAnimatedIndoorParts.fruitTypeIcons) do
				if fruitTypeIcon.node ~= nil then
					setVisibility(fruitTypeIcon.node, currentFruitType ~= FillType.UNKNOWN);
				
					if currentFruitType ~= FillType.UNKNOWN then
						local oldMaterial = getMaterial(fruitTypeIcon.node, 0);
						local newMaterial = g_materialManager:getMaterial(currentFruitType, "icon", 1);

						setVisibility(fruitTypeIcon.node, newMaterial ~= nil);
					
						if newMaterial ~= nil then
							ConfigurationUtil.replaceMaterialRec(self, fruitTypeIcon.node, oldMaterial, newMaterial);
						end;
					end;
				end;
			end;
		end;

		if specAnimatedIndoorParts.driveSymbols ~= nil then
			for _, driveSymbol in pairs(specAnimatedIndoorParts.driveSymbols) do
				if driveSymbol.name == "DISABLE_ON_MOTOR_OFF" then
					setVisibility(driveSymbol.node, not self:getIsMotorStarted());
				end;
				
				if self:getIsMotorStarted() then
					if driveSymbol.name ~= "CONTROL_LAMP" and driveSymbol.name ~= "DISABLE_ON_MOTOR_OFF" then
						setShaderParameter(driveSymbol.node, "lightControl", 0, 0, 0, 0, false);
						setVisibility(driveSymbol.node, true);
					end;
					
					if driveSymbol.name == "FOUR_WHEEL" then
						if fourWheel then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "DIFF_LOCK_FRONT" then
						if diffLockFront then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "DIFF_LOCK_BACK" then
						if diffLockBack then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "PTO_FRONT" then
						if isPTOFrontActive then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "PTO_BACK" then
						if isPTOBackActive then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TIPPING_ACTIVE" then
						if trailerIsTipping then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
						
					elseif driveSymbol.name == "TIPPING_INACTIVE" then
						if numberOfAttachedTrailers > 0 and not trailerIsTipping then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
						
					elseif driveSymbol.name == "TIPPING_SIDE_LEFT_1" then
						setVisibility(driveSymbol.node, isLeftTipping1);
						
						if isLeftTipping1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "TIPPING_SIDE_RIGHT_1" then
						setVisibility(driveSymbol.node, isRightTipping1);
						
						if isRightTipping1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "TIPPING_SIDE_BACK_1" then
						setVisibility(driveSymbol.node, isBackTipping1);
						
						if isBackTipping1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
						
					elseif driveSymbol.name == "TIPPING_SIDE_GRAINDOOR_1" then
						setVisibility(driveSymbol.node, isGrainDoorTipping1);

						if isGrainDoorTipping1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
						
					elseif driveSymbol.name == "TIPPING_SIDE_LEFT_2" then
						setVisibility(driveSymbol.node, isLeftTipping2);
						
						if isLeftTipping2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "TIPPING_SIDE_RIGHT_2" then
						setVisibility(driveSymbol.node, isRightTipping2);
						
						if isRightTipping2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "TIPPING_SIDE_BACK_2" then
						setVisibility(driveSymbol.node, isBackTipping2);
						
						if isBackTipping2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
						
					elseif driveSymbol.name == "TIPPING_SIDE_GRAINDOOR_2" then
						setVisibility(driveSymbol.node, isGrainDoorTipping2);
						
						if isGrainDoorTipping2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "TRAILER_1" then
						if numberOfAttachedTrailers >= 1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "TRAILER_2" then
						if numberOfAttachedTrailers >= 2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_LOWERED" then
						if moveDownFront then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_BACK_LOWERED" then
						if moveDownBack then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_LIFTED" then
						if not moveDownFront then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_BACK_LIFTED" then
						if not moveDownBack then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "HANDBRAKE" then
						if handbrakeIsActive then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "CONTROL_LAMP" then
						if driveSymbol.currentTime < driveSymbol.displayTime then 
							setVisibility(driveSymbol.node, true);
						end;
						
						if driveSymbol.currentTime < (driveSymbol.displayTime + driveSymbol.displayTime) and driveSymbol.currentTime > driveSymbol.displayTime then 
							setVisibility(driveSymbol.node, false);
						end;

						driveSymbol.currentTime = driveSymbol.currentTime + dt;
					
					elseif driveSymbol.name == "GPS_STEERING_ACTIVE" then
						if gpsSteeringIsActive then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "GPS_ACTIVE" then
						if gpsIsActive then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "GPS_LANE_PLUS" then
						setVisibility(driveSymbol.node, gpsCurrentLane > 0);

					elseif driveSymbol.name == "GPS_LANE_MINUS" then
						setVisibility(driveSymbol.node, gpsCurrentLane < 0);
					
					elseif driveSymbol.name == "STEERAXLE_IS_LOCKED" then
						if steerAxleIsLocked then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "STEERAXLE_IS_UNLOCKED" then
						setVisibility(driveSymbol.node, false);

					elseif driveSymbol.name == "RIDGE_MARKER_LEFT" then
						if ridgeMarkerState == 1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "RIDGE_MARKER_RIGHT" then
						if ridgeMarkerState == 2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TIPPING_ACTIVE_TRUCK" then
						if truckIsTipping then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_IS_UNFOLDED" then
						if toolFrontIsUnfolded then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_IS_FOLDED" then
						if not toolFrontIsUnfolded and hasFoldablePartsFront then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_BACK_IS_UNFOLDED" then
						if toolBackIsUnfolded then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TOOL_BACK_IS_FOLDED" then
						if not toolBackIsUnfolded and hasFoldablePartsBack then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "ABS_TRUCK" then
						if isTruckABSActive then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "ABS_TRAILER" then
						if isTruckABSActive and numberOfAttachedTrailers >= 1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "TIPPING_ACTIVE_TRAILER_1" then
						if numberOfAttachedTrailers >= 1 and trailerIsTipping1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
						
					elseif driveSymbol.name == "TIPPING_ACTIVE_TRAILER_2" then
						if numberOfAttachedTrailers >= 2 and trailerIsTipping2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "WEARED_OUT_BRAKES" then
						if brakesWearedOut then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "WARNING_BRAKE_COMPRESSOR_FILL_LEVEL_TO_LOW" then
						if airCompressorDoRefill then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "WARNING_WORKSHOP_1" then
						if warningWorkshop1 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "WARNING_WORKSHOP_2" then
						if warningWorkshop2 then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "STOP_IMMINENTLY" then
						if stopImminently then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "MOTOR_NEED_MAINTRACE" then
						if motorNeedMaintrace then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
					
					elseif driveSymbol.name == "LOW_OIL_PRESSURE" then
						if lowOilPressure then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;
						
					elseif driveSymbol.name == "OIL_FILL_LEVEL_TO_LOW" then
						if oilFillLevelToLow then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "VEHICLE_SELECTED" then
						setVisibility(driveSymbol.node, drivableIsSelected);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[1] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.1, 0, 1, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_SELECTED_1" then
						setVisibility(driveSymbol.node, toolFrontIsSelected1);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[2] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.1, 0, 1, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_SELECTED_2" then
						setVisibility(driveSymbol.node, toolFrontIsSelected2);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						
					elseif driveSymbol.name == "TOOL_BACK_SELECTED_1" then
						setVisibility(driveSymbol.node, toolBackIsSelected1);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[3] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.1, 0, 1, false);
						end;

					elseif driveSymbol.name == "TOOL_BACK_SELECTED_2" then
						setVisibility(driveSymbol.node, toolBackIsSelected2);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[4] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.1, 0, 1, false);
						end;

					elseif driveSymbol.name == "VEHICLE_UNSELECTED" then
						setVisibility(driveSymbol.node, true);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[1] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.25, 0, 1, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_UNSELECTED_1" then
						setVisibility(driveSymbol.node, toolFrontIsAttached1);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[2] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.25, 0, 1, false);
						end;

					elseif driveSymbol.name == "TOOL_FRONT_UNSELECTED_2" then
						setVisibility(driveSymbol.node, toolFrontIsAttached2);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);

					elseif driveSymbol.name == "TOOL_BACK_UNSELECTED_1" then
						setVisibility(driveSymbol.node, toolBackIsAttached1);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[3] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.25, 0, 1, false);
						end;

					elseif driveSymbol.name == "TOOL_BACK_UNSELECTED_2" then
						setVisibility(driveSymbol.node, toolBackIsAttached2);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
						if vehicleSchemaIsTurnedOn[4] then
							setShaderParameter(driveSymbol.node, "emitColor", 0, 0.25, 0, 1, false);
						end;

					elseif driveSymbol.name == "FRONTLOADER_SELECTED" then
						setVisibility(driveSymbol.node, frontloaderIsSelected);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
					elseif driveSymbol.name == "FRONTLOADER_UNSELECTED" then
						setVisibility(driveSymbol.node, frontloaderIsAttached);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
					elseif driveSymbol.name == "FRONTLOADER_TOOL_SELECTED" then
						setVisibility(driveSymbol.node, frontloaderToolIsSelected);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);
						
					elseif driveSymbol.name == "FRONTLOADER_TOOL_UNSELECTED" then
						setVisibility(driveSymbol.node, frontloaderToolIsAttached);

						setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						setShaderParameter(driveSymbol.node, "emitColor", driveSymbol.emitColor[1], driveSymbol.emitColor[2], driveSymbol.emitColor[3], driveSymbol.emitColor[4], false);

					elseif driveSymbol.name == "STRAW_CHOPPER" then
						if isStrawChopperActive then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "OVERLOADING_ACTIVE" then
						if isOverloading then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "GRAIN_TANK_UNFOLDED" then
						if isGrainTankUnfolded then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "PIPE_UNFOLDED" then
						if isPipeUnfolded then
							setShaderParameter(driveSymbol.node, "lightControl", driveSymbol.intensity, 0, 0, 0, false);
						end;

					elseif driveSymbol.name == "RIDGE_MARKER_LEFT_UP" then
						setVisibility(driveSymbol.node, ridgeMarkerState ~= 1);
					
					elseif driveSymbol.name == "RIDGE_MARKER_RIGHT_UP" then
						setVisibility(driveSymbol.node, ridgeMarkerState ~= 2);
					
					elseif driveSymbol.name == "RIDGE_MARKER_LEFT_DOWN" then
						setVisibility(driveSymbol.node, ridgeMarkerState == 1);
						
					elseif driveSymbol.name == "RIDGE_MARKER_RIGHT_DOWN" then
						setVisibility(driveSymbol.node, ridgeMarkerState == 2);	

					elseif driveSymbol.name == "CARDINAL_DIRECTION_NORTH" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 0 and specAnimatedIndoorParts.cardinalDirection < 45);

					elseif driveSymbol.name == "CARDINAL_DIRECTION_NORTH_EAST" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 45 and specAnimatedIndoorParts.cardinalDirection < 90);
						
					elseif driveSymbol.name == "CARDINAL_DIRECTION_EAST" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 90 and specAnimatedIndoorParts.cardinalDirection < 125);

					elseif driveSymbol.name == "CARDINAL_DIRECTION_SOUTH_EAST" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 125 and specAnimatedIndoorParts.cardinalDirection < 170);

					elseif driveSymbol.name == "CARDINAL_DIRECTION_SOUTH" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 170 and specAnimatedIndoorParts.cardinalDirection < 205);

					elseif driveSymbol.name == "CARDINAL_DIRECTION_SOUTH_WEST" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 205 and specAnimatedIndoorParts.cardinalDirection < 250);

					elseif driveSymbol.name == "CARDINAL_DIRECTION_WEST" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 250 and specAnimatedIndoorParts.cardinalDirection < 295);

					elseif driveSymbol.name == "CARDINAL_DIRECTION_NORTH_WEST" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.cardinalDirection >= 295 and specAnimatedIndoorParts.cardinalDirection < 360);

					elseif driveSymbol.name == "NO_FIELD" then
						setVisibility(driveSymbol.node, specAnimatedIndoorParts.currentFieldNumber == 0);

					elseif driveSymbol.name == "PS_MODE_AUTO" then
						setVisibility(driveSymbol.node, currentModeIsAuto);

					elseif driveSymbol.name == "PS_MODE_SEMI" then
						setVisibility(driveSymbol.node, currentModeIsSemi);

					elseif driveSymbol.name == "PS_MODE_MANUAL" then
						setVisibility(driveSymbol.node, currentModeIsManual);
						
					elseif driveSymbol.name == "PS_LANE_LEFT" then
						setVisibility(driveSymbol.node, true);
						
						local currentLightControl = 0;
							
						if currentShutoffModeIsRight then
							currentLightControl = driveSymbol.intensity;
						end;
							
						setShaderParameter(driveSymbol.node, "lightControl", currentLightControl, 0, 0, 0, false);
						
					elseif driveSymbol.name == "PS_LANE_RIGHT" then
						setVisibility(driveSymbol.node, true);
						
						local currentLightControl = 0;
							
						if currentShutoffModeIsLeft then
							currentLightControl = driveSymbol.intensity;
						end;
							
						setShaderParameter(driveSymbol.node, "lightControl", currentLightControl, 0, 0, 0, false);

					elseif driveSymbol.name == "PS_TRAM_LANE_LEFT" then
						setVisibility(driveSymbol.node, true);
						
						local currentLightControl = 0;
							
						if createTramLines or currentShutoffModeIsRight then
							currentLightControl = driveSymbol.intensity;
						end;
							
						setShaderParameter(driveSymbol.node, "lightControl", currentLightControl, 0, 0, 0, false);
						
					elseif driveSymbol.name == "PS_TRAM_LANE_RIGHT" then
						setVisibility(driveSymbol.node, true);
						
						local currentLightControl = 0;
							
						if createTramLines or currentShutoffModeIsLeft then
							currentLightControl = driveSymbol.intensity;
						end;
							
						setShaderParameter(driveSymbol.node, "lightControl", currentLightControl, 0, 0, 0, false);
					end;
				else
					if driveSymbol.name ~= "DISABLE_ON_MOTOR_OFF" then	
						setVisibility(driveSymbol.node, false);
					end;
					
					if driveSymbol.name == "CONTROL_LAMP" then
						driveSymbol.currentTime = driveSymbol.intensity;
					end;
				end;

				local intensity = 0;

				local turnLightDirections = {
					"ANY",
					"LEFT",
					"RIGHT"
				};

				for _, turnLightDirection in pairs(turnLightDirections) do
					for trailerNumber = 1, 2 do
						if driveSymbol.name == "TURN_LIGHT_TRAILER_" .. turnLightDirection .. "_" .. trailerNumber then
							setShaderParameter(driveSymbol.node, "lightControl", 0, 0, 0, 0, false);

							if numberOfAttachedTrailersWithTurnLights >= trailerNumber then		
								if turnLightDirection == "ANY" then	
									if self.spec_lights.turnLightState ~= Lights.TURNLIGHT_OFF then
										intensity = driveSymbol.intensity;
									end;
								else
									if self.spec_lights.turnLightState == Lights["TURNLIGHT_" .. turnLightDirection] or self.spec_lights.turnLightState == Lights.TURNLIGHT_HAZARD then
										intensity = driveSymbol.intensity;
									end;
								end;
							end;
	
							setShaderParameter(driveSymbol.node, "lightControl", intensity, 0, 0, 0, false);
						end;
					end;
				end;
			end;
		end;
	end;

	if self:getIsActive() then
		local camIsInside = false;

		if self.isClient and isActiveForInputIgnoreSelection and self.spec_enterable.cameras ~= nil then
			for cameraNumber, camera in pairs(self.spec_enterable.cameras) do
				if camera.isInside and camera.isActivated then
					local newFov = math.deg(getFovY(camera.cameraNode));
					local allowUpdateFov = false;

					if specAnimatedIndoorParts.inputIsPressed then
						if newFov > 23 then
							newFov = math.max(newFov - 0.055 * dt, 23);

							allowUpdateFov = true;
						end;
					else
						if newFov < camera.oldFovSaved then	
							newFov = math.min(newFov + 0.055 * dt, camera.oldFovSaved);

							allowUpdateFov = true;
						end;
					end;

					if allowUpdateFov then
						setFovY(camera.cameraNode, math.rad(newFov));
					end;

					if specAnimatedIndoorParts.resetIndoorCamera then
						camera:resetCamera();
					end;

					camIsInside = true;
				end;
			end;
		end;

		for _, input in pairs({"INDOOR_CAMERA_ZOOM_BUTTON", "INDOOR_CAMERA_RESET_BUTTON"}) do
			local inputButton = specAnimatedIndoorParts.actionEvents[InputAction[input]];

			if inputButton ~= nil then
				g_inputBinding:setActionEventTextVisibility(inputButton.actionEventId, camIsInside);
				g_inputBinding:setActionEventActive(inputButton.actionEventId, camIsInside);
			end;
		end;

		specAnimatedIndoorParts.inputIsPressed = false;
		specAnimatedIndoorParts.resetIndoorCamera = false;
	end;
end;

function AnimatedIndoorParts:getFillLevelPercentTotal()
	local percentage = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then	
		local fillLevel = 0;
		local capacity = 0;

		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");

			if specFillUnit ~= nil then
				for _, fillUnit in pairs(specFillUnit.fillUnits) do
					fillLevel = fillLevel + fillUnit.fillLevel;
					capacity = capacity + fillUnit.capacity;
				end;
			end;

			local specAttachJoints = object.spec_attacherJoints;

			for attachedImplement = 1, #specAttachJoints.attachedImplements do
				local object = specAttachJoints.attachedImplements[attachedImplement].object;
				local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");
	
				if specFillUnit ~= nil then
					for _, fillUnit in pairs(specFillUnit.fillUnits) do
						fillLevel = fillLevel + fillUnit.fillLevel;
						capacity = capacity + fillUnit.capacity;
					end;
				end;
			end;
		end;

		if fillLevel > 0 then
			percentage = math.floor((fillLevel / capacity) * 100);
		end;
	end;

	return percentage;
end;

function AnimatedIndoorParts:getFillLevelTotal()
	local fillLevel = 0;
	
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");

			if specFillUnit ~= nil then
				for _, fillUnit in pairs(specFillUnit.fillUnits) do
					fillLevel = fillLevel + fillUnit.fillLevel;
				end;
			end;

			local specAttachJoints = object.spec_attacherJoints;
			
			for attachedImplement = 1, #specAttachJoints.attachedImplements do
				local object = specAttachJoints.attachedImplements[attachedImplement].object;
				local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");
	
				if specFillUnit ~= nil then
					for _, fillUnit in pairs(specFillUnit.fillUnits) do
						fillLevel = fillLevel + fillUnit.fillLevel;
					end;
				end;
			end;
		end;
	end;

	return fillLevel;
end;

function AnimatedIndoorParts:getFillLevelPercentTrailer1()
	local percentage = 0;
	
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local fillLevel = 0;
		local capacity = 0;

		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");

			if specFillUnit ~= nil then
				for _, fillUnit in pairs(specFillUnit.fillUnits) do
					fillLevel = fillLevel + fillUnit.fillLevel;
					capacity = capacity + fillUnit.capacity;
				end;
			end;
		end;

		if fillLevel > 0 then
			percentage = math.floor((fillLevel / capacity) * 100);
		end;
	end;

	return percentage;
end;

function AnimatedIndoorParts:getFillLevelTrailer1()
	local fillLevel = 0;
	
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then	
		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");

			if specFillUnit ~= nil then
				for _, fillUnit in pairs(specFillUnit.fillUnits) do
					fillLevel = fillLevel + fillUnit.fillLevel;
				end;
			end;
		end;
	end;

	return fillLevel;
end;

function AnimatedIndoorParts:getFillLevelPercentTrailer2()
	local percentage = 0;
	
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local fillLevel = 0;
		local capacity = 0;

		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specAttachJoints = object.spec_attacherJoints;

			for attachedImplement = 1, #specAttachJoints.attachedImplements do
				local object = specAttachJoints.attachedImplements[attachedImplement].object;
				local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");
	
				if specFillUnit ~= nil then
					for _, fillUnit in pairs(specFillUnit.fillUnits) do
						fillLevel = fillLevel + fillUnit.fillLevel;
						capacity = capacity + fillUnit.capacity;
					end;
				end;
			end;
		end;

		if fillLevel > 0 then
			percentage = math.floor((fillLevel / capacity) * 100);
		end;
	end;

	return percentage;
end;

function AnimatedIndoorParts:getFillLevelTrailer2()
	local fillLevel = 0;
	
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then	
		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specAttachJoints = object.spec_attacherJoints;
			
			for attachedImplement = 1, #specAttachJoints.attachedImplements do
				local object = specAttachJoints.attachedImplements[attachedImplement].object;
				local specFillUnit = JohnDeerePackUtil.getSpecByName(object, "fillUnit");
	
				if specFillUnit ~= nil then
					for _, fillUnit in pairs(specFillUnit.fillUnits) do
						fillLevel = fillLevel + fillUnit.fillLevel;
					end;
				end;
			end;
		end;
	end;

	return fillLevel;
end;

function AnimatedIndoorParts:getGpsWorkingWidth()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local gpsWorkingWidth = 0;

	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specGlobalPositioningSystem = JohnDeerePackUtil.getSpecByName(self, "globalPositioningSystem", specAnimatedIndoorParts.gpsModName);

		if specGlobalPositioningSystem ~= nil and specGlobalPositioningSystem.guidanceData ~= nil then
			gpsWorkingWidth = specGlobalPositioningSystem.guidanceData.width;
		end;
	end;

	return gpsWorkingWidth;
end;

function AnimatedIndoorParts:getGpsCurrentLane()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local currentLane = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specGlobalPositioningSystem = JohnDeerePackUtil.getSpecByName(self, "globalPositioningSystem", specAnimatedIndoorParts.gpsModName);

		if specGlobalPositioningSystem ~= nil and specGlobalPositioningSystem.guidanceData ~= nil then
			currentLane = specGlobalPositioningSystem.guidanceData.currentLane;

			if currentLane < 0 then
				currentLane = -currentLane;
			end;
		end;
	end;

	return currentLane;
end;

function AnimatedIndoorParts:getCurrentBaleCount()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local currentBaleCount = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specBaleCounter = JohnDeerePackUtil.getSpecByName(object, "baleCounter");

			if specBaleCounter ~= nil then
				currentBaleCount = specBaleCounter.countToday;
			end;
		end;
	end;

	return currentBaleCount;
end;

function AnimatedIndoorParts:getTotalBaleCount()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local totalBaleCount = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specBaleCounter = JohnDeerePackUtil.getSpecByName(object, "baleCounter");

			if specBaleCounter ~= nil then
				totalBaleCount = specBaleCounter.countTotal;
			end;
		end;
	end;

	return totalBaleCount;
end;

function AnimatedIndoorParts:getCurrentWrappedBaleCount()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local currentWrappedBaleCount = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specWrappedBaleCounter = JohnDeerePackUtil.getSpecByName(object, "wrappedBaleCounter");

			if specWrappedBaleCounter  ~= nil then
				currentWrappedBaleCount = specWrappedBaleCounter.countToday;
			end;
		end;
	end;

	return currentWrappedBaleCount;
end;

function AnimatedIndoorParts:getTotalWrappedBaleCount()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local totalWrappedBaleCount = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;
		
		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specWrappedBaleCounter = JohnDeerePackUtil.getSpecByName(object, "wrappedBaleCounter");

			if specWrappedBaleCounter ~= nil then
				totalWrappedBaleCount = specWrappedBaleCounter.countTotal;
			end;
		end;
	end;

	return totalWrappedBaleCount;
end;

function AnimatedIndoorParts:getMotorLoad()
	local motorLoad = Utils.getNoNil(self.GUIMotorLoad, 0);

	return motorLoad;
end;

function AnimatedIndoorParts:getWheelSlip()
	local wheelSlip = Utils.getNoNil(self.GUISlip, 0);

	return wheelSlip;
end;

function AnimatedIndoorParts:getFuelUsage()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local specMotorized = JohnDeerePackUtil.getSpecByName(self, "motorized");
	--[[local currentFuelUsage = 0;
	local dt = specAnimatedIndoorParts.dt;

	if specMotorized ~= nil and specMotorized.isMotorStarted then
		local idleFactor = 0.5;
		local rpmPercentage = (specMotorized.motor:getLastMotorRpm() - specMotorized.motor:getMinRpm()) / (specMotorized.motor:getMaxRpm() - specMotorized.motor:getMinRpm());
		local rpmFactor = idleFactor + rpmPercentage * (1-idleFactor);
		local loadFactor = specMotorized.smoothedLoadPercentage * rpmPercentage;
		local motorFactor = 0.5 * ((0.2*rpmFactor) + (1.8*loadFactor));
		local usageFactor = 1.0;

		if g_currentMission.missionInfo.fuelUsageLow then
			usageFactor = 0.7;
		end;

		local damage = self:getVehicleDamage();

		if damage > 0 then
			usageFactor = usageFactor * (1 + damage * Motorized.DAMAGED_USAGE_INCREASE)
		end;
		
		for _,consumer in pairs(specMotorized.consumers) do
			if consumer.permanentConsumption and consumer.usage > 0 then
				local used = usageFactor * motorFactor * consumer.usage * dt;

				if used ~= 0 then
					local fillType = self:getFillUnitLastValidFillType(consumer.fillUnitIndex);

					if fillType == FillType.DIESEL then
						currentFuelUsage = used / dt * 1000 * 60 * 60 -- per hour
					end;
				end;
			end;
		end;
	end;]]
	
	return specMotorized.lastFuelUsageDisplay;
end;

function AnimatedIndoorParts:getMotorTemperature()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local specMotorized = JohnDeerePackUtil.getSpecByName(self, "motorized");
	--[[local currentMotorTemperature = 0;
	local dt = specAnimatedIndoorParts.dt;

    local delta = specMotorized.motorTemperature.heatingPerMS * dt;
	local factor = (1 + 4 * specMotorized.actualLoadPercentage) / 5;
	
	delta = delta * (factor + self:getMotorRpmPercentage());
	
	specMotorized.motorTemperature.value = math.min(specMotorized.motorTemperature.valueMax, specMotorized.motorTemperature.value + delta);
	
    -- cooling due to wind
    delta = specMotorized.motorTemperature.coolingByWindPerMS * dt;
	
	local speedFactor = math.pow(math.min(1.0, self:getLastSpeed() / 30), 2);

	specMotorized.motorTemperature.value = math.max(specMotorized.motorTemperature.valueMin, specMotorized.motorTemperature.value - (speedFactor * delta));
	
    -- cooling per fan
    if specMotorized.motorTemperature.value > specMotorized.motorFan.enableTemperature then
        specMotorized.motorFan.enabled = true;
	end;
	
    if specMotorized.motorFan.enabled then
        if specMotorized.motorTemperature.value < specMotorized.motorFan.disableTemperature then
            specMotorized.motorFan.enabled = false;
        end;
	end;
	
    if specMotorized.motorFan.enabled then
		local delta = specMotorized.motorFan.coolingPerMS * dt;
		
        specMotorized.motorTemperature.value = math.max(specMotorized.motorTemperature.valueMin, specMotorized.motorTemperature.value - delta);
	end;]]
	
	return specMotorized.motorTemperature.value;
end;

function AnimatedIndoorParts:getPsCurrentLane()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local currentLane = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;

		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);

			if specProSeedTramLines ~= nil then
				currentLane = specProSeedTramLines.currentLane;
			else
				local specAttachJoints = object.spec_attacherJoints;

				for attachedImplement = 1, #specAttachJoints.attachedImplements do
					local object = specAttachJoints.attachedImplements[attachedImplement].object;
					local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
				
					if specProSeedTramLines ~= nil then
						currentLane = specProSeedTramLines.currentLane;
					else
						local specAttachJoints = object.spec_attacherJoints;

						for attachedImplement = 1, #specAttachJoints.attachedImplements do
							local object = specAttachJoints.attachedImplements[attachedImplement].object;
							local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
						
							if specProSeedTramLines ~= nil then
								currentLane = specProSeedTramLines.currentLane;
							end;
						end;
					end;
				end;
			end;
		end;
	end;

	return currentLane;
end;

function AnimatedIndoorParts:getPsMaxLanes()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local maxLanes = 1;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;

		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);

			if specProSeedTramLines ~= nil then
				maxLanes = specProSeedTramLines.tramLineDistance / specProSeedTramLines.workingWidthRounded;
			else
				local specAttachJoints = object.spec_attacherJoints;

				for attachedImplement = 1, #specAttachJoints.attachedImplements do
					local object = specAttachJoints.attachedImplements[attachedImplement].object;
					local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
				
					if specProSeedTramLines ~= nil then
						maxLanes = specProSeedTramLines.tramLineDistance / specProSeedTramLines.workingWidthRounded;
					else
						local specAttachJoints = object.spec_attacherJoints;

						for attachedImplement = 1, #specAttachJoints.attachedImplements do
							local object = specAttachJoints.attachedImplements[attachedImplement].object;
							local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
						
							if specProSeedTramLines ~= nil then
								maxLanes = specProSeedTramLines.tramLineDistance / specProSeedTramLines.workingWidthRounded;
							end;
						end;
					end;
				end;
			end;
		end;
	end;

	return maxLanes;
end;

function AnimatedIndoorParts:getPsWorkingWidth()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local currentWorkingWidth = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;

		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);

			if specProSeedTramLines ~= nil then
				currentWorkingWidth = specProSeedTramLines.workingWidth;
			else
				local specAttachJoints = object.spec_attacherJoints;

				for attachedImplement = 1, #specAttachJoints.attachedImplements do
					local object = specAttachJoints.attachedImplements[attachedImplement].object;
					local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
				
					if specProSeedTramLines ~= nil then
						currentWorkingWidth = specProSeedTramLines.workingWidth;
					else
						local specAttachJoints = object.spec_attacherJoints;

						for attachedImplement = 1, #specAttachJoints.attachedImplements do
							local object = specAttachJoints.attachedImplements[attachedImplement].object;
							local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
						
							if specProSeedTramLines ~= nil then
								currentWorkingWidth = specProSeedTramLines.workingWidth;
							end;
						end;
					end;
				end;
			end;
		end;
	end;

	return currentWorkingWidth;
end;

function AnimatedIndoorParts:getPsTramlineDistance()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local currentTramlineDistance = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local specAttachJoints = self.spec_attacherJoints;

		for attachedImplement = 1, #specAttachJoints.attachedImplements do
			local object = specAttachJoints.attachedImplements[attachedImplement].object;
			local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);

			if specProSeedTramLines ~= nil then
				currentTramlineDistance = specProSeedTramLines.tramLineDistance;
			else
				local specAttachJoints = object.spec_attacherJoints;

				for attachedImplement = 1, #specAttachJoints.attachedImplements do
					local object = specAttachJoints.attachedImplements[attachedImplement].object;
					local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
				
					if specProSeedTramLines ~= nil then
						currentTramlineDistance = specProSeedTramLines.tramLineDistance;
					else
						local specAttachJoints = object.spec_attacherJoints;

						for attachedImplement = 1, #specAttachJoints.attachedImplements do
							local object = specAttachJoints.attachedImplements[attachedImplement].object;
							local specProSeedTramLines = JohnDeerePackUtil.getSpecByName(object, "proSeedTramLines", specAnimatedIndoorParts.psModName);
						
							if specProSeedTramLines ~= nil then
								currentTramlineDistance = specProSeedTramLines.tramLineDistance;
							end;
						end;
					end;
				end;
			end;
		end;
	end;

	return currentTramlineDistance;
end;

function AnimatedIndoorParts:getCurrentFieldNumber()
	local specAnimatedIndoorParts = JohnDeerePackUtil.getSpecByName(self, "animatedIndoorParts");
	local currentFieldNumber = 0;
	local isEnteredAsPassenger = false;

	if g_universalPassenger ~= nil then
        isEnteredAsPassenger = g_universalPassenger.currentPassengerVehicle ~= nil;
    end;
	
	if self:getIsActive() or isEnteredAsPassenger then
		local posX, posZ;

    	if g_currentMission.controlledVehicle ~= nil then
    	    posX, _, posZ = getWorldTranslation(g_currentMission.controlledVehicle.rootNode);
    	elseif g_currentMission.player ~= nil then
    	    posX, _, posZ, _ = g_currentMission.player:getPositionData();
    	end;

		local farmland = g_farmlandManager:getFarmlandAtWorldPosition(posX, posZ);

    	if farmland ~= nil then
			local fieldMapping = g_fieldManager.farmlandIdFieldMapping[farmland.id];

    	    if fieldMapping ~= nil and fieldMapping[1] ~= nil then
    	        currentFieldNumber = fieldMapping[1].fieldId;
    	    end;
		end;

		for _, dashboards in pairs(self.spec_dashboard.dashboards) do
			if dashboards.valueFunc ~= nil and dashboards.valueFunc == "getCurrentFieldNumber" then	
				if currentFieldNumber == 0 then
					--## fix to disable the numbers if the field number is 0
					I3DUtil.setNumberShaderByValue(dashboards.numbers, 0, dashboards.precision, false);
				end;
			end;
		end;
	end;
	
	specAnimatedIndoorParts.currentFieldNumber = currentFieldNumber;

	return currentFieldNumber;
end;

-----------------------------------------------------------------------------
--## Multiplayer Event
-----------------------------------------------------------------------------

AnimatedIndoorPartsScreenEvent = {};
AnimatedIndoorPartsScreenEvent_mt = Class(AnimatedIndoorPartsScreenEvent, Event);

InitEventClass(AnimatedIndoorPartsScreenEvent, "AnimatedIndoorPartsScreenEvent");

function AnimatedIndoorPartsScreenEvent:emptyNew()
	local self = Event:new(AnimatedIndoorPartsScreenEvent_mt);
    
	return self;
end;

function AnimatedIndoorPartsScreenEvent:new(tractor, currentScreenNumber)
	local self = AnimatedIndoorPartsScreenEvent:emptyNew();
	
	self.tractor = tractor;
	self.currentScreenNumber = currentScreenNumber;

	return self;
end;

function AnimatedIndoorPartsScreenEvent:readStream(streamId, connection)
	self.tractor = NetworkUtil.readNodeObject(streamId);
	self.currentScreenNumber = streamReadInt16(streamId);
	
	self:run(connection);
end;

function AnimatedIndoorPartsScreenEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.tractor);

	streamWriteInt16(streamId, self.currentScreenNumber);
end;

function AnimatedIndoorPartsScreenEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(AnimatedIndoorPartsScreenEvent:new(self.tractor, self.currentScreenNumber), nil, connection, self.tractor);
	end;
	
    if self.tractor ~= nil then
        self.tractor:setCurrentScreen(self.currentScreenNumber, true);
	end;
end;