----------------------------------------------------------------------------------------------------
-- MovingSheets
----------------------------------------------------------------------------------------------------
-- Purpose: Specialization for sheet animation.
--
-- Copyright (c) Wopster, 2020
----------------------------------------------------------------------------------------------------

---@class MovingSheets
MovingSheets = {}
MovingSheets.MOD_NAME = g_currentModName

MovingSheets.SPEED_MAX_IMPACT = 7
MovingSheets.MOVEMENT_SCALE = 20

MovingSheets.FACTOR_CURVE = AnimCurve:new(linearInterpolator1)
MovingSheets.FACTOR_CURVE:addKeyframe({ 0, time = 0 })
MovingSheets.FACTOR_CURVE:addKeyframe({ 0, time = 0.1 })
MovingSheets.FACTOR_CURVE:addKeyframe({ 1, time = 1 })

function MovingSheets.prerequisitesPresent(specializations)
    return true
end

function MovingSheets.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "loadSheetFromXML", MovingSheets.loadSheetFromXML)
    SpecializationUtil.registerFunction(vehicleType, "getImpactFactor", MovingSheets.getImpactFactor)
    SpecializationUtil.registerFunction(vehicleType, "updateSheet", MovingSheets.updateSheet)
    SpecializationUtil.registerFunction(vehicleType, "resetSheets", MovingSheets.resetSheets)
    SpecializationUtil.registerFunction(vehicleType, "getNextActiveSheet", MovingSheets.getNextActiveSheet)
    SpecializationUtil.registerFunction(vehicleType, "getMowerLoad", MovingSheets.getMowerLoad)
end

function MovingSheets.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", MovingSheets)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", MovingSheets)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", MovingSheets)
    SpecializationUtil.registerEventListener(vehicleType, "onPostAttach", MovingSheets)
    SpecializationUtil.registerEventListener(vehicleType, "onPostDetach", MovingSheets)
end

---Called on load.
function MovingSheets:onLoad(savegame)
    self.spec_movingSheets = self[("spec_%s.movingSheets"):format(MovingSheets.MOD_NAME)]
    local spec = self.spec_movingSheets

    spec.lastImpactFactor = 0
    spec.isActive = false

    spec.sheetIndex = 1

    spec.sheets = {}
    local i = 0
    while true do
        local key = ("vehicle.movingSheets.sheet(%d)"):format(i)

        if not hasXMLProperty(self.xmlFile, key) then
            break
        end

        local entry = {}
        if self:loadSheetFromXML(entry, self.xmlFile, key) then
            table.insert(spec.sheets, entry)
        end

        i = i + 1
    end
end

---Called on post load.
function MovingSheets:onPostLoad(savegame)
    self:resetSheets(true)
end

---Called on update frame.
function MovingSheets:onUpdate(dt)
    local spec = self.spec_movingSheets

    if self.isClient and self.firstTimeRun and spec.isActive then
        local impactFactor = self:getImpactFactor()
        local doReset = math.abs(spec.lastImpactFactor) < 1
        local forceUpdate = doReset

        -- If we are moving constant speed we force update.
        if math.abs(impactFactor) > 0 and math.abs(impactFactor - spec.lastImpactFactor) <= 0.005 then
            forceUpdate = true
        end

        for _, sheet in ipairs(spec.sheets) do
            if sheet.isActive then
                self:updateSheet(sheet, impactFactor, doReset, forceUpdate)
            end
        end

        spec.lastImpactFactor = impactFactor
    end
end

---Gets the next active sheet.
function MovingSheets:getNextActiveSheet(sheetIndex)
    local spec = self.spec_movingSheets

    local next = sheetIndex + 1
    if next > #spec.sheets then
        next = 1
    end

    local sheet = spec.sheets[next]
    if not sheet.isActive then
        return self:getNextActiveSheet(next)
    end

    return sheet, next
end

---Called on post attach.
function MovingSheets:onPostAttach(attacherVehicle, inputJointDescIndex, jointDescIndex)
    self:resetSheets(true)
end

---Called on post detach.
function MovingSheets:onPostDetach(attacherVehicle, implement)
    self:resetSheets(false)
end

--------------
-- Functions
--------------

---Returns the calculated impact factor ranging from -1 to 1.
function MovingSheets:getImpactFactor()
    local lastSpeed = self:getLastSpeed()
    local speedFactor = math.max(0, lastSpeed / MovingSheets.SPEED_MAX_IMPACT) * 10
    local movingDirection = lastSpeed < 1 and 0 or self.movingDirection
    return speedFactor * movingDirection
end

---Returns the mower load when the mower is present, can be used as a load function for a sheet.
function MovingSheets:getMowerLoad()
    if self.spec_mower ~= nil then
        return self.spec_mower.workAreaParameters.lastUsedAreasPct
    end

    return 0
end

---Updates sheet deformation based on the given factor.
function MovingSheets:updateSheet(sheet, factor, reset, force)
    reset = reset or false
    force = force or false

    local absFactor = math.abs(factor)
    local movementTime = MovingSheets.MOVEMENT_SCALE * 0.001 * absFactor * sheet.movementScale

    sheet.currentUpdateTime = sheet.currentUpdateTime + movementTime

    if math.abs(factor - sheet.factor) > 0.01 or force then
        sheet.factor = factor

        local direction = MathUtil.sign(factor)
        local movementCurveFactor = MovingSheets.FACTOR_CURVE:get(movementTime)
        local movementFactor = MathUtil.clamp(movementCurveFactor * direction, -sheet.movementMax, sheet.movementMax)

        if sheet.isFoldable then
            local foldAnimationTime = self:getFoldAnimTime()

            if foldAnimationTime ~= sheet.currentFoldAnimationTime or reset then
                sheet.currentFoldAnimationTime = math.min(foldAnimationTime, sheet.foldingMaxTime) * sheet.foldingScale
                local xCompression, yCompression, zCompression, _ = getShaderParameter(sheet.node, sheet.shaderParameterDeformerName)
                setShaderParameter(sheet.node, sheet.shaderParameterDeformerName, xCompression, yCompression, zCompression, sheet.currentFoldAnimationTime * sheet.foldingDirection, false)
            end
        end

        if not sheet.allowsMovementImpact then
            movementFactor = 0
        end

        if sheet.loadFunction ~= nil then
            local load = sheet.loadFunction(self, sheet) * 0.5
            if load ~= 0 then
                movementFactor = MathUtil.clamp(movementFactor + (load * direction), -sheet.movementMax, sheet.movementMax)

                -- Increase amplitude on the sheet when it has a load.
                sheet.currentUpdateTime = sheet.currentUpdateTime + load * 1.25
            end
        end

        local shaderParameters = { getShaderParameter(sheet.node, sheet.shaderParameterName) }

        if sheet.resetOnZero then
            shaderParameters[sheet.shaderParameterComponentAmplitude] = reset and 0 or sheet.shaderParameter[sheet.shaderParameterComponentAmplitude]
        end

        sheet.currentImpact = movementFactor
        shaderParameters[sheet.shaderParameterComponentImpact] = movementFactor

        shaderParameters[sheet.shaderParameterComponentTime] = sheet.currentUpdateTime

        local x, y, z, w = unpack(shaderParameters)
        setShaderParameter(sheet.node, sheet.shaderParameterName, x, y, z, w, false)
    end
end

---Resets the sheets to not move actively.
function MovingSheets:resetSheets(isActive)
    local spec = self.spec_movingSheets
    spec.isActive = isActive

    if self.isClient then
        for _, sheet in ipairs(spec.sheets) do
            sheet.isActive = isActive and getVisibility(sheet.node)
            self:updateSheet(sheet, 0, true, true)
        end
    end
end

---Loads the sheet from the xml and validates setup.
---Returns true when valid, false otherwise.
function MovingSheets:loadSheetFromXML(sheet, xmlFile, baseKey)
    local node = I3DUtil.indexToObject(self.components, getXMLString(xmlFile, baseKey .. "#node"), self.i3dMappings)

    if node ~= nil then
        sheet.node = node
        sheet.isActive = getVisibility(sheet.node) -- don't update hidden sheets e.g. with configurations.

        sheet.shaderParameterName = Utils.getNoNil(getXMLString(xmlFile, baseKey .. "#shaderParameterName"), "sheet")
        sheet.shaderParameterDeformerName = Utils.getNoNil(getXMLString(xmlFile, baseKey .. "#shaderParameterDeformerName"), "sheetDeformer")

        if not getHasShaderParameter(sheet.node, sheet.shaderParameterName)
            or not getHasShaderParameter(sheet.node, sheet.shaderParameterDeformerName) then
            g_logManager:xmlWarning(self.configFileName, "Shader parameters on node '%s' not found. Please set the correct shader variation!", getName(sheet.node))
            return false
        end

        sheet.factor = 0
        sheet.currentUpdateTime = 0
        sheet.currentImpact = 0
        sheet.currentFoldAnimationTime = 0

        sheet.functionName = getXMLString(xmlFile, baseKey .. "#functionName")
        if sheet.functionName ~= nil then
            if self[sheet.functionName] == nil then
                g_logManager:xmlWarning(self.configFileName, "Given sheet functionName '%s' not defined. Please add missing function or specialization!", tostring(sheet.functionName))
                return false
            end

            sheet.loadFunction = self[sheet.functionName]
        end

        local function boolToInt(boolean, signed)
            return signed and (boolean and 1 or -1) or (boolean and 1 or 0)
        end

        sheet.allowsMovementImpact = Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#allowsMovementImpact"), true)
        sheet.isFlippable = Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#isFlippable"), false)
        sheet.isRight = Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#isRight"), false)
        sheet.resetOnZero = Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#resetOnZero"), false)

        sheet.allowFoldYAxis = Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#allowFoldYAxis"), false)
        sheet.allowFoldZAxis = Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#allowFoldZAxis"), true)

        sheet.isFoldable = Utils.getNoNil(getXMLBool(xmlFile, baseKey .. "#isFoldable"), false)
        sheet.foldingMaxTime = Utils.getNoNil(getXMLFloat(xmlFile, baseKey .. "#foldingMaxTime"), 1)
        sheet.foldingScale = Utils.getNoNil(getXMLFloat(xmlFile, baseKey .. "#foldingScale"), 1)
        sheet.foldingDirection = boolToInt(sheet.isRight, true)

        if sheet.isFoldable then
            sheet.currentFoldAnimationTime = math.min(self:getFoldAnimTime(), sheet.foldingMaxTime)
        end

        sheet.movementMax = Utils.getNoNil(getXMLFloat(xmlFile, baseKey .. "#movementMax"), 1)
        sheet.movementScale = Utils.getNoNil(getXMLFloat(xmlFile, baseKey .. "#movementScale"), 1)

        sheet.shaderParameterComponentTime = Utils.getNoNil(getXMLInt(xmlFile, baseKey .. "#shaderParameterComponentTime"), 2)
        sheet.shaderParameterComponentImpact = Utils.getNoNil(getXMLInt(xmlFile, baseKey .. "#shaderParameterComponentImpact"), 3)
        sheet.shaderParameterComponentAmplitude = Utils.getNoNil(getXMLInt(xmlFile, baseKey .. "#shaderParameterComponentAmplitude"), 4)
        local x, y, z, w = StringUtil.getVectorFromString(Utils.getNoNil(getXMLString(xmlFile, baseKey .. "#shaderParameter"), "0 0 0 0"))
        sheet.shaderParameter = { x, y, z, w }
        setShaderParameter(node, sheet.shaderParameterName, x, y, z, w, false)

        local xCompression = boolToInt(sheet.isFlippable)
        local yCompression = boolToInt(sheet.allowFoldYAxis)
        local zCompression = boolToInt(sheet.allowFoldZAxis)
        local foldTime = sheet.currentFoldAnimationTime * sheet.foldingDirection
        setShaderParameter(node, sheet.shaderParameterDeformerName, xCompression, yCompression, zCompression, foldTime, false)

        return true
    end

    return false
end
