----------------------------------------------------------------------------------------------------
-- SectionLift
----------------------------------------------------------------------------------------------------
-- Purpose: Specialization for controlling section lifting.
--
-- Copyright (c) Wopster, 2020
----------------------------------------------------------------------------------------------------

---@class SectionLift
SectionLift = {}
SectionLift.MOD_NAME = g_currentModName
SectionLift.SPEED_THRESHOLD = 7
SectionLift.LITERS_PER_SECTION = 9
SectionLift.MOVE_TIME_MULTIPLIER = 0.3 -- speed up move time.
SectionLift.COOL_DOWN_THRESHOLD = 1000 -- ms

function SectionLift.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(WorkArea, specializations) and SpecializationUtil.hasSpecialization(Foldable, specializations)
end

function SectionLift.initSpecialization()
    g_configurationManager:addConfigurationType("sectionLift", g_i18n:getText("configuration_sectionLift"), nil, nil, nil, nil, ConfigurationUtil.SELECTOR_MULTIOPTION)
end

function SectionLift.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "loadSectionFromXML", SectionLift.loadSectionFromXML)
    SpecializationUtil.registerFunction(vehicleType, "setSectionDirection", SectionLift.setSectionDirection)
    SpecializationUtil.registerFunction(vehicleType, "isSectionLowered", SectionLift.isSectionLowered)
    SpecializationUtil.registerFunction(vehicleType, "getSectionFoldDirection", SectionLift.getSectionFoldDirection)
    SpecializationUtil.registerFunction(vehicleType, "handleLowering", SectionLift.handleLowering)
    SpecializationUtil.registerFunction(vehicleType, "hasSectionFruitContact", SectionLift.hasSectionFruitContact)
    SpecializationUtil.registerFunction(vehicleType, "setSectionLiftActive", SectionLift.setSectionLiftActive)
    SpecializationUtil.registerFunction(vehicleType, "isSectionLiftAllowed", SectionLift.isSectionLiftAllowed)
end

function SectionLift.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", SectionLift)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", SectionLift)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", SectionLift)
    SpecializationUtil.registerEventListener(vehicleType, "onFoldStateChanged", SectionLift)
    SpecializationUtil.registerEventListener(vehicleType, "onAIImplementStart", SectionLift)
end

function SectionLift.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "doCheckSpeedLimit", SectionLift.doCheckSpeedLimit)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getLoweringActionEventState", SectionLift.getLoweringActionEventState)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getIsFoldMiddleAllowed", SectionLift.getIsFoldMiddleAllowed)
end

---Called on load.
function SectionLift:onLoad(savegame)
    self.spec_sectionLift = self[("spec_%s.sectionLift"):format(SectionLift.MOD_NAME)]
    local spec = self.spec_sectionLift

    local configurationId = Utils.getNoNil(self.configurations["sectionLift"], 1)
    local baseKey = ("vehicle.sectionLiftConfigurations.sectionLiftConfiguration(%d)"):format(configurationId - 1)

    spec.isActive = false
    spec.allowsSectionLift = Utils.getNoNil(getXMLBool(self.xmlFile, baseKey .. "#allowsSectionLift"), true)

    spec.isMower = SpecializationUtil.hasSpecialization(Mower, self.specializations)

    spec.foldMiddleDirection = self.spec_foldable.foldMiddleDirection
    spec.foldMiddleAnimationTime = self.spec_foldable.foldMiddleAnimTime
    spec.turnOnFoldDirection = self.spec_foldable.turnOnFoldDirection

    spec.allSectionsHighered = false
    spec.allSectionsLowered = false
    spec.factorMultiplier = Utils.getNoNil(getXMLFloat(self.xmlFile, baseKey .. "#factorMultiplier"), 1)

    spec.sections = {}
    local i = 0
    while true do
        local key = ("%s.sections.section(%d)"):format(baseKey, i)

        if not hasXMLProperty(self.xmlFile, key) then
            break
        end

        local entry = {}
        if self:loadSectionFromXML(entry, self.xmlFile, key) then
            entry.id = i + 1
            table.insert(spec.sections, entry)
        end

        i = i + 1
    end

    spec.litersLimit = SectionLift.LITERS_PER_SECTION * #spec.sections
end

---Called on update frame.
function SectionLift:onUpdateTick(dt)
    local spec = self.spec_sectionLift

    if not spec.isActive or not spec.allowsSectionLift then
        return
    end

    local allSectionsLowered = true
    local isLowered = self:getIsLowered()
    local lastSpeed = self:getLastSpeed()
    local speedFactor = math.pow(math.max(1, lastSpeed / SectionLift.SPEED_THRESHOLD), 2)

    for id, section in ipairs(spec.sections) do
        if self.isServer then
            section.cooldownTime = section.cooldownTime + dt

            if section.cooldownTime >= SectionLift.COOL_DOWN_THRESHOLD then
                local direction = self:getSectionFoldDirection(section, lastSpeed, speedFactor)
                if direction ~= 0 then
                    self:setSectionDirection(id, direction, direction > 0)
                    section.cooldownTime = 0
                end
            end
        end

        if section.foldingPartIndex ~= nil and math.abs(section.direction) > 0.1 then
            local isInvalid = false
            local isSectionLowered = section.direction < -0.1
            local foldAnimTime = isSectionLowered and 1 or 0

            local foldingPart = self.spec_foldable.foldingParts[section.foldingPartIndex]
            local animationTime = self:getRealAnimationTime(foldingPart.animationName)
            local normAnimationTime = animationTime / self.spec_foldable.maxFoldAnimDuration

            -- can never be 0 as we eliminated that with the abs check, so else is always < 0.
            if section.direction > 0 then
                if animationTime < foldingPart.animDuration then
                    isInvalid = true
                end

                foldAnimTime = math.max(foldAnimTime, normAnimationTime)
            else
                if animationTime > 0 then
                    isInvalid = true
                end

                foldAnimTime = math.min(foldAnimTime, normAnimationTime)
            end

            section.foldAnimationTime = MathUtil.clamp(foldAnimTime, 0, 1)

            if isInvalid and self.isServer then
                if foldingPart.componentJoint ~= nil then
                    self:setComponentJointFrame(foldingPart.componentJoint, foldingPart.anchorActor)
                end
            end

            allSectionsLowered = allSectionsLowered and self:isSectionLowered(section)
        end
    end

    if #spec.sections > 1 then
        spec.allSectionsLowered = allSectionsLowered

        if not spec.allSectionsLowered and isLowered then
            self:handleLowering(false)
        elseif spec.allSectionsLowered and not isLowered then
            self:handleLowering(true)
        end
    end
end

---Get the next direction to fold for the section.
function SectionLift:getSectionFoldDirection(section, speed, speedFactor)
    local isLowered = section.direction < -0.1 or self:getIsLowered()
    local workArea = self.spec_workArea.workAreas[section.workAreaIndex]
    local hasFruitContact = self.movingDirection > 0 and speed > 0.5 and self:hasSectionFruitContact(section, workArea, speedFactor)
    local doHigher = (section.direction == -1 or isLowered) and not hasFruitContact
    local doLower = (section.direction == 1 or not isLowered) and hasFruitContact

    local direction = 0
    if doHigher or doLower then
        direction = doHigher and 1 or -1
    end

    return direction
end

---Returns true whenever the section has fruit contact, false otherwise.
function SectionLift:hasSectionFruitContact(section, workArea, speedFactor)
    local spec = self.spec_sectionLift
    local x0, _, z0 = getWorldTranslation(workArea.start)
    local x1, _, z1 = getWorldTranslation(workArea.width)

    local centerX = (x0 + x1) * 0.5
    local centerZ = (z0 + z1) * 0.5

    local farmId = self:getActiveFarm() or AccessHandler.EVERYONE
    local isAccessible, _ = self:getIsAccessibleAtWorldPosition(farmId, centerX, centerZ, workArea.type)

    if not isAccessible then
        return false
    end

    local x2, _, z2 = getWorldTranslation(workArea.height)
    local dx, _, dz = localDirectionToWorld(workArea.width, 0, 0, 1)
    local offset = section.workAreaSize * speedFactor * spec.factorMultiplier

    x0 = x0 + offset * dx
    x1 = x1 + offset * dx
    x2 = x2 + offset * dx
    z0 = z0 + offset * dz
    z1 = z1 + offset * dz
    z2 = z2 + offset * dz

    -- Use center position from center to area start to detect if we're still on the field.
    local sideX = (x0 + ((x0 + x1) * 0.5)) * 0.5
    local sideZ = (z0 + ((z0 + z1) * 0.5)) * 0.5

    local isOnField = getDensityAtWorldPos(g_currentMission.terrainDetailId, sideX, 0, sideZ) ~= 0
    if isOnField then
        isOnField = getDensityAtWorldPos(g_currentMission.terrainDetailId, x1, 0, z1) ~= 0
    end

    if not isOnField then
        return false
    end

    local fruits = g_fruitTypeManager:getFruitTypes()
    if workArea.lastValidPickupFruitType ~= FruitType.UNKNOWN then
        fruits = {
            [workArea.lastValidPickupFruitType or FruitType.UNKNOWN] = {}
        }
    end

    if spec.isMower then
        fruits = self.spec_mower.fruitTypeConverters
    end

    for fruitType, _ in pairs(fruits) do
        if spec.isMower then
            local fruitValue, _, _, _ = FSDensityMapUtil.getFruitArea(fruitType, x0, z0, x1, z1, x2, z2, nil, true)
            if fruitValue > 0 then
                return true
            end
        else
            local fillTypeIndex = g_fruitTypeManager:getWindrowFillTypeIndexByFruitTypeIndex(fruitType)
            if fillTypeIndex ~= FillType.UNKNOWN then

                local fillType = g_fillTypeManager:getFillTypeByIndex(fillTypeIndex)
                if fillType ~= nil then
                    local liters, _, _ = DensityMapHeightUtil.getFillLevelAtArea(fillTypeIndex, x0, z0, x1, z1, x2, z2)
                    if liters > 0 and liters < spec.litersLimit then
                        return true
                    end
                end
            end
        end
    end

    return false
end

function SectionLift:loadSectionFromXML(section, xmlFile, baseKey)
    section.workAreaIndex = Utils.getNoNil(getXMLInt(xmlFile, baseKey .. "#workAreaIndex"), 1)
    section.foldingPartIndex = getXMLInt(xmlFile, baseKey .. "#foldingPartIndex")
    section.foldAnimationTime = 0
    section.direction = 1 -- initial highered.
    section.cooldownTime = 0

    local workArea = self.spec_workArea.workAreas[section.workAreaIndex]
    if workArea ~= nil then
        section.workAreaSize = calcDistanceFrom(workArea.start, workArea.height)
    else
        -- Todo: error
        return false
    end

    return true
end

function SectionLift:setSectionLiftActive(isActive, noEventSend)
    local spec = self.spec_sectionLift

    if spec.isActive ~= isActive then
        SectionLiftEvent.sendEvent(self, isActive, noEventSend)

        local actionEvent = spec.actionEvents[InputAction.SECTION_LIFT_KP]
        if actionEvent ~= nil then
            local key = isActive and "action_disableSectionLift" or "action_enableSectionLift"
            g_inputBinding:setActionEventText(actionEvent.actionEventId, g_i18n:getText(key))
        end

        if self.isServer then
            local attacherVehicle = self:getAttacherVehicle()
            local jointDesc = attacherVehicle:getAttacherJointDescFromObject(self)

            if isActive then
                -- Lower first
                local attacherJointIndex = attacherVehicle:getAttacherJointIndexFromObject(self)
                self:setLoweredAll(true, attacherJointIndex)

                if jointDesc.moveTimeBackup == nil then
                    jointDesc.moveTimeBackup = jointDesc.moveTime
                end

                jointDesc.moveTime = jointDesc.moveTimeBackup * SectionLift.MOVE_TIME_MULTIPLIER

                for id, section in ipairs(spec.sections) do
                    local workArea = self.spec_workArea.workAreas[section.workAreaIndex]

                    if workArea ~= nil then
                        local direction = self:getSectionFoldDirection(section, 1, 1)
                        if direction ~= 0 then
                            self:setSectionDirection(id, direction, direction > 0, true)
                        end
                    end
                end
            else
                jointDesc.moveTime = jointDesc.moveTimeBackup
            end
        end

        --Set state here so we can do the initial setup correctly.
        spec.isActive = isActive
    end
end

function SectionLift:isSectionLiftAllowed()
    return self:getIsInWorkPosition()
end

function SectionLift:handleLowering(doLowering)
    local spec = self.spec_sectionLift

    if self.getAttacherVehicle == nil or not self.isServer then
        return
    end

    local attacherVehicle = self:getAttacherVehicle()
    local attacherJointIndex = attacherVehicle:getAttacherJointIndexFromObject(self)

    -- Sync folding time with the first section.
    if table.getn(self.spec_foldable.foldingParts) > 0 then
        if attacherJointIndex ~= nil then
            local attacherJoints = attacherVehicle:getAttacherJoints()
            local attacherJoint = attacherJoints[attacherJointIndex]
            attacherJoint.moveDown = doLowering
        end

        local section = spec.sections[1]
        self.spec_foldable.foldAnimTime = section.foldAnimationTime
        self.spec_foldable.foldMoveDirection = section.direction
        self.spec_foldable.moveToMiddle = section.moveToMiddle
    end

    attacherVehicle:handleLowerImplementByAttacherJointIndex(attacherJointIndex, doLowering)
end

function SectionLift:isSectionLowered(section)
    local middleAnimationTime = self.spec_foldable.foldMiddleAnimTime
    local foldMiddleDirection = self.spec_foldable.foldMiddleDirection

    local spec = self.spec_sectionLift
    if spec.isActive and middleAnimationTime ~= nil then
        if section.direction ~= 0 then
            if foldMiddleDirection > 0 then
                if section.foldAnimationTime < middleAnimationTime + 0.01 then
                    return section.direction < 0 and section.moveToMiddle ~= true
                end
            elseif section.foldAnimationTime > middleAnimationTime - 0.01 then
                return section.direction > 0 and section.moveToMiddle ~= true
            end
        elseif foldMiddleDirection > 0 and section.foldAnimationTime < 0.01 then
            return true
        elseif foldMiddleDirection < 0 and math.abs(1 - section.foldAnimationTime) < 0.01 then
            return true
        end
    end

    return false
end

function SectionLift:setSectionDirection(id, direction, moveToMiddle, force, noEventSend)
    force = force or false

    local spec_foldable = self.spec_foldable
    local section = self.spec_sectionLift.sections[id]

    if section ~= nil and section.direction ~= direction or force then
        SectionLiftSectionEvent.sendEvent(self, id, direction, moveToMiddle, force, noEventSend)

        section.direction = direction

        if section.foldingPartIndex ~= nil then
            local foldingPart = spec_foldable.foldingParts[section.foldingPartIndex]

            if section.moveToMiddle ~= moveToMiddle or force then
                section.moveToMiddle = moveToMiddle

                local middleAnimationTime = spec_foldable.foldMiddleAnimTime
                local isLowered = direction < -0.1
                local isRaised = direction > 0.1
                local sign = MathUtil.sign(section.direction)

                local speedScale
                if isLowered or isRaised then
                    if not section.moveToMiddle
                        or ((isLowered and middleAnimationTime < section.foldAnimationTime)
                        or (isRaised and middleAnimationTime > section.foldAnimationTime)) then
                        speedScale = foldingPart.speedScale * sign
                    end
                end

                self:stopAnimation(foldingPart.animationName, true)

                if speedScale ~= nil then
                    local animationTime
                    if self:getIsAnimationPlaying(foldingPart.animationName) then
                        animationTime = self:getAnimationTime(foldingPart.animationName)
                    else
                        animationTime = section.foldAnimationTime * spec_foldable.maxFoldAnimDuration / self:getAnimationDuration(foldingPart.animationName)
                    end

                    self:playAnimation(foldingPart.animationName, speedScale, animationTime, true)

                    if section.moveToMiddle then
                        local stopAnimationTime = middleAnimationTime * spec_foldable.maxFoldAnimDuration / self:getAnimationDuration(foldingPart.animationName)
                        self:setAnimationStopTime(foldingPart.animationName, stopAnimationTime)
                    end
                end

                if isLowered or isRaised then
                    section.foldAnimationTime = MathUtil.clamp(section.foldAnimationTime + (0.0001 * sign), 0, 1)
                end
            end
        else
            self:handleLowering(not (direction > 0))
        end
    end
end


function SectionLift:onFoldStateChanged(direction, moveToMiddle)
    local spec = self.spec_sectionLift

    if self.isClient then
        if spec.actionEvents ~= nil then
            local actionEvent = spec.actionEvents[InputAction.SECTION_LIFT_KP]
            if actionEvent ~= nil then
                g_inputBinding:setActionEventActive(actionEvent.actionEventId, direction < 0)
            end
        end
    end
end

function SectionLift:onAIImplementStart()
    self:setSectionLiftActive(false)
end

---------------------
-- Injections
---------------------

function SectionLift:doCheckSpeedLimit(superFunc)
    local spec = self.spec_sectionLift

    if superFunc(self) then
        return true
    end

    if spec.isActive then
        for _, section in ipairs(spec.sections) do
            local isLowered = section.direction < -0.1
            if isLowered then
                return true
            end
        end
    end

    return false
end

function SectionLift:getLoweringActionEventState(superFunc)
    local spec = self.spec_sectionLift
    if spec.isActive then
        return false
    end

    return superFunc(self)
end

function SectionLift:getIsFoldMiddleAllowed(superFunc)
    local spec = self.spec_sectionLift
    if spec.isActive then
        return false
    end

    return superFunc(self)
end

---------------------
-- ActionEvents
---------------------

function SectionLift.actionEventToggleSectionLift(self, ...)
    local spec = self.spec_sectionLift
    if spec.allowsSectionLift then
        local toActiveState = not spec.isActive
        if self.getAttacherVehicle ~= nil then
            local attacherVehicle = self:getAttacherVehicle()

            for _, implement in pairs(attacherVehicle:getAttachedImplements()) do
                local object = implement.object

                if object ~= nil and object.setSectionLiftActive ~= nil and object.spec_sectionLift.allowsSectionLift then
                    object:setSectionLiftActive(toActiveState)
                end
            end
        else
            self:setSectionLiftActive(toActiveState)
        end
    end
end

function SectionLift:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if self.isClient then
        local spec = self.spec_sectionLift
        self:clearActionEventsTable(spec.actionEvents)

        if spec.allowsSectionLift and isActiveForInputIgnoreSelection then
            local _, actionEventToggleId = self:addActionEvent(spec.actionEvents, InputAction.SECTION_LIFT_KP, self, SectionLift.actionEventToggleSectionLift, false, true, false, true, nil)

            local key = spec.isActive and "action_disableSectionLift" or "action_enableSectionLift"
            g_inputBinding:setActionEventText(actionEventToggleId, g_i18n:getText(key))
            g_inputBinding:setActionEventTextPriority(actionEventToggleId, GS_PRIO_NORMAL)
            g_inputBinding:setActionEventActive(actionEventToggleId, self:isSectionLiftAllowed())
        end
    end
end
