----------------------------------------------------------------------------------------------------
-- SaltSpreader
----------------------------------------------------------------------------------------------------
-- Purpose:  Allows spreading salt to melt snow.
--
-- Copyright (c) Realismus Modding, 2019
----------------------------------------------------------------------------------------------------

---@class SaltSpreader
SaltSpreader = {}

---Returns true when the requirements are present, false otherwise.
---@param specializations table
---@return boolean
function SaltSpreader.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Sprayer, specializations)
end

function SaltSpreader.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "processSaltSpreaderArea", SaltSpreader.processSaltSpreaderArea)
end

function SaltSpreader.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getSprayerUsage", SaltSpreader.getSprayerUsage)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCanToggleTurnedOn", SaltSpreader.getCanToggleTurnedOn)
end

function SaltSpreader.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", SaltSpreader)
end

---Called on load.
---@param savegame table
function SaltSpreader:onLoad(savegame)
    local spec = self.spec_sprayer
    spec.useSpeedLimit = false -- We don't want to limit speed while driving on the road.
end

---Processes the salt spreading and call the seasons snow handler.
---@param workArea table
---@param dt number
---@return number changedArea
---@return number totalArea
function SaltSpreader:processSaltSpreaderArea(workArea, dt)
    local spec = self.spec_sprayer

    if spec.workAreaParameters.sprayFillLevel <= 0 then
        return 0, 0
    end

    if g_seasons == nil then
        return 0, 0
    end

    local snowHandler = g_seasons.snowHandler
    local startWorldX, _, startWorldZ = getWorldTranslation(workArea.start)
    local widthWorldX, _, widthWorldZ = getWorldTranslation(workArea.width)
    local heightWorldX, _, heightWorldZ = getWorldTranslation(workArea.height)

    local changedArea, totalArea = snowHandler:saltArea(startWorldX, startWorldZ, widthWorldX, widthWorldZ, heightWorldX, heightWorldZ)

    spec.workAreaParameters.isActive = true
    spec.workAreaParameters.lastChangedArea = spec.workAreaParameters.lastChangedArea + changedArea
    spec.workAreaParameters.lastStatsArea = spec.workAreaParameters.lastStatsArea + changedArea
    spec.workAreaParameters.lastTotalArea = spec.workAreaParameters.lastTotalArea + totalArea

    if self:getLastSpeed() > 1 then
        spec.isWorking = true
    end

    return changedArea, totalArea
end

---Usage depends on vehicle speed (automatic adjustment)
function SaltSpreader:getSprayerUsage(superFunc, fillType, dt)
    if fillType == FillType.UNKNOWN then
        return 0
    end

    local spec = self.spec_sprayer
    local scale = Utils.getNoNil(spec.usageScale.fillTypeScales[fillType], spec.usageScale.default)
    local litersPerSecond = 1
    local sprayType = g_sprayTypeManager:getSprayTypeByFillTypeIndex(fillType)
    if sprayType ~= nil then
        litersPerSecond = sprayType.litersPerSecond
    end

    local activeSprayType = self:getActiveSprayType()
    local workWidth = spec.usageScale.workingWidth
    if activeSprayType ~= nil then
        workWidth = activeSprayType.usageScale.workingWidth or workWidth
    end

    return scale * litersPerSecond * math.max(self.speedLimit, self:getLastSpeed()) * workWidth * dt * 0.001
end

---Do not allow turning on when Seasons is not available.
function SaltSpreader:getCanToggleTurnedOn(superFunc)
    return superFunc(self) and g_seasons ~= nil
end
