--[[

Game:	 Farming Simulator 19
Script:  ToggleAnimatedParts
Author:  ThundR (original by sven777b)
Version: 1.0.0.0

-----------------------------------------------------------------------------------------------------------------------------------------------------
Example vehicle.xml:
The names used below are all just examples, you can use whatever names you like for vehicleType, actionId, and actionText

<vehicle type="tractorAnimParts">

	<animParts>
  		<animPart node="0>1" 	actionId="ROTATE_PART"	actionText="action_rotatePart" 	rotMin="0 0 0" 	 	rotMax="15 0 0" 	moveTime="3" showHelp="true"/>
  		<animPart node="0>2" 	actionId="MOVE_PART"   	actionText="Move Part" 		 	transMin="0 0 0" 	transMax="0 0.4 0" 	moveTime="3" showHelp="true"/>
  		<animPart node="0>3|0" 	actionId="SCALE_PART" 									scaleMin="1 0.5 1" 	scaleMax="1 1 1" 	moveTime="3"/>
  		<animPart node="0>3|1" 	actionId="SPIN_PART"	actionText="action_spinPart" 	permRotAxis="3" 						moveTime="0.1"/>

  		<animPart node= 0>5" 	 toggleVisibility="true" 				listenTo="motorized.isMotorStarted" reverseListen="true"/>
  		<animPart node="0>4" 	 permRotAxis="2" 		moveTime="0.1"	listenTo="beaconLightsActive"/>
	</animParts>

</vehicle>

-----------------------------------------------------------------------------------------------------------------------------------------------------
Legend:

- node: 			Index of animated part
- actionId:			Action id of animated part (optional)
- actionText:		Action text (optional, uses actionId text if omitted)
- rotMin/Max:		Minimum and maximum rotation for animation part (optional)
- transMin/Max: 	Minimum and maximum translation for animated part (optional)
- scaleMin/Max:		Minimum and maximum scale for animated part (optional)
- toggleVisibility: Shows/hides animation when active/inactive
- moveTime:			Speed of animation (seconds)
- showHelp:			Show action help text in help window (optional)
- isMovingTool: 	Animated part is also a moving tool (optional, needed to make dependent parts move together)
- listenTo:			Variable that, if true, activates the animated part (optional)
- reverseListen:	Activates the animated part on false instead of true (optional)
	* To specify a specialization, separate the listenTo string with a "." (i.e. turnOnVehicle.isTurnedOn)

-----------------------------------------------------------------------------------------------------------------------------------------------------
Example modDesc.xml
The names and texts below are all just examples. You can use whatever binds and names you like, they just have to match the vehicle.xml

    <l10n>
		<text name="input_ROTATE_PART">
			<en>My Vehicle: Rotate Part</en>
		</text>
		<text name="input_MOVE_PART">
			<en>My Vehicle: Move Part</en>
		</text>
		<text name="input_SCALE_PART">
			<en>My Vehicle: Scale Part</en>
		</text>
		<text name="input_SPIN_PART">
			<en>My Vehicle: Spin Part</en>
		</text>

		<text name="action_rotatePart">
			<en>Rotate part</en>
		</text>
		<text name="action_spinPart">
			<en>Spin part</en>
		</text>
    </l10n>

    <actions>
        <action name="ROTATE_PART" axisType="HALF"/>
        <action name="MOVE_PART" axisType="HALF"/>
        <action name="SCALE_PART" axisType="HALF"/>
        <action name="SPIN_PART" axisType="HALF"/>
    </actions>

    <inputBinding>
        <actionBinding action="ROTATE_PART">
            <binding device="KB_MOUSE_DEFAULT" input="KEY_o"/>
        </actionBinding>
        <actionBinding action="MOVE_PART">
            <binding device="KB_MOUSE_DEFAULT" input="KEY_l"/>
        </actionBinding>
        <actionBinding action="SCALE_PART">
            <binding device="KB_MOUSE_DEFAULT" input="KEY_u"/>
        </actionBinding>
        <actionBinding action="SPIN_PART">
            <binding device="KB_MOUSE_DEFAULT" input="KEY_j"/>
        </actionBinding>
	</inputBinding>

    <specializations>
        <specialization name="toggleAnimParts" className="ToggleAnimatedParts" filename="scripts/ToggleAnimatedParts.lua"/>
    </specializations>

	<vehicleTypes>
		<type name="tractorAnimParts" parent="tractor" className="Vehicle" filename="$dataS/scripts/vehicles/Vehicle.lua">
            <specialization name="toggleAnimParts"/>
        </type>
	</vehicleTypes>

--]]

local g_modName = g_currentModName
local g_specName = string.format("spec_%s.toggleAnimParts", g_modName)

ToggleAnimatedParts = {}

function ToggleAnimatedParts.prerequisitesPresent(specializations)
    return true
end

function ToggleAnimatedParts.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onLoad", ToggleAnimatedParts)
	SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", ToggleAnimatedParts)
	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", ToggleAnimatedParts)
	SpecializationUtil.registerEventListener(vehicleType, "onReadStream", ToggleAnimatedParts)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", ToggleAnimatedParts)
end

-- [Event Functions] ********************************************************************************************************************************

function ToggleAnimatedParts:onLoad(savegame)
	local spec = self[g_specName]
	spec.animParts = {}

    local idx = 0
    while true do
        local baseString = string.format("vehicle.animParts.animPart(%d)#", idx)
        local part = {}

		part.node 		= I3DUtil.indexToObject(self.components, getXMLString(self.xmlFile, baseString.."node"), self.i3dMappings)
        part.actionId	= getXMLString(self.xmlFile,  baseString.."actionId")
        part.listenTo	= getXMLString(self.xmlFile,  baseString.."listenTo")

        if part.node == nil or (InputAction[part.actionId] == nil and part.listenTo == nil) then
            break
        end

		part.index = #spec.animParts + 1

		if part.listenTo ~= nil then
			local entries = StringUtil.splitString(".", part.listenTo)
			if #entries > 1 then
				part.listenSpec = string.format("spec_%s", entries[1])
				part.listenTo	= entries[2]
			end
			part.reverseListen = Utils.getNoNil(getXMLBool(self.xmlFile, baseString.."reverseListen"), false)
		end

        part.showHelp = Utils.getNoNil(getXMLBool(self.xmlFile, baseString.."showHelp"), false)
		if part.showHelp then
			part.actionText = getXMLString(self.xmlFile,  baseString.."actionText")
		end

        local rotMin = getXMLString(self.xmlFile, baseString.."rotMin")
        local rotMax = getXMLString(self.xmlFile, baseString.."rotMax")
        if rotMin ~= nil and rotMax ~= nil then
            local x,y,z = StringUtil.getVectorFromString(rotMin)
            part.rotMin = {math.rad(Utils.getNoNil(x,0)), math.rad(Utils.getNoNil(y,0)), math.rad(Utils.getNoNil(z,0))}
            local x,y,z = StringUtil.getVectorFromString(rotMax)
            part.rotMax = {math.rad(Utils.getNoNil(x,0)), math.rad(Utils.getNoNil(y,0)), math.rad(Utils.getNoNil(z,0))}
        end

        local transMin = getXMLString(self.xmlFile, baseString.."transMin")
        local transMax = getXMLString(self.xmlFile, baseString.."transMax")
        if transMin ~= nil and transMax ~= nil then
            local x,y,z = StringUtil.getVectorFromString(transMin)
            part.transMin = {Utils.getNoNil(x,0), Utils.getNoNil(y,0), Utils.getNoNil(z,0)}
            local x,y,z = StringUtil.getVectorFromString(transMax)
            part.transMax = {Utils.getNoNil(x,0), Utils.getNoNil(y,0), Utils.getNoNil(z,0)}
        end

        local scaleMin = getXMLString(self.xmlFile, baseString.."scaleMin")
        local scaleMax = getXMLString(self.xmlFile, baseString.."scaleMax")
        if scaleMin ~= nil and scaleMax ~= nil then
            local x,y,z = StringUtil.getVectorFromString(scaleMin)
            part.scaleMin = {Utils.getNoNil(x,1), Utils.getNoNil(y,1), Utils.getNoNil(z,1)}
            local x,y,z = StringUtil.getVectorFromString(scaleMax)
            part.scaleMax = {Utils.getNoNil(x,1), Utils.getNoNil(y,1), Utils.getNoNil(z,1)}
        end

        local permRotAxis = getXMLInt(self.xmlFile, baseString.."permRotAxis")
        if permRotAxis ~= nil then
            if permRotAxis >= 1 and permRotAxis <= 3 then
                part.permRotAxis = permRotAxis
            end
        end

        part.toggleVisibility	= Utils.getNoNil(getXMLBool(self.xmlFile, baseString.."toggleVisibility"), false)
		setVisibility(part.node, not part.toggleVisibility)

        part.moveTime 			= Utils.getNoNil(getXMLFloat(self.xmlFile, baseString.."moveTime"), 1) * 1000
        part.autoReturn 		= Utils.getNoNil(getXMLBool(self.xmlFile, baseString.."autoReturn"), false)
        part.isMovingTool 		= Utils.getNoNil(getXMLBool(self.xmlFile, baseString.."isMovingTool"), false)
        part.state 				= false
		part.lastState			= false
		part.isMoving			= false

        table.insert(spec.animParts, part)
        idx = idx + 1
    end
end

function ToggleAnimatedParts:onUpdateTick(dt)
    if self:getIsActive() then
		local spec = self[g_specName]

        for _, animPart in ipairs(spec.animParts) do
            if animPart.listenTo ~= nil then
				local listenState = self[animPart.listenTo]
				if animPart.listenSpec and type(self[animPart.listenSpec]) == "table" then
					listenState = self[animPart.listenSpec][animPart.listenTo]
				end

				if type(listenState) == "boolean" then
					if animPart.reverseListen then
						animPart.state = not listenState
					else
                		animPart.state = listenState
					end
                end
            end

			if animPart.lastState ~= animPart.state then
				animPart.isMoving = true
				animPart.lastState = animPart.state
			end

            if animPart.isMoving then
                animPart.isMoving = false

				if animPart.toggleVisibility then
					setVisibility(animPart.node, animPart.state)
				end

                if animPart.rotMin ~= nil then
                    local curRot = {getRotation(animPart.node)}
                    local newRot = Utils.getMovedLimitedValues(curRot, animPart.rotMax, animPart.rotMin, 3, animPart.moveTime, dt, not animPart.state)
                    setRotation(animPart.node, unpack(newRot))
                    for i = 1, 3 do
                        if math.abs(newRot[i]-curRot[i]) > 0.001 then
                            animPart.isMoving = true
                        end
                    end
                end

                if animPart.transMin ~= nil then
                    local curTrans = {getTranslation(animPart.node)}
                    local newTrans = Utils.getMovedLimitedValues(curTrans, animPart.transMax, animPart.transMin, 3, animPart.moveTime, dt, not animPart.state)
                    setTranslation(animPart.node, unpack(newTrans))
                    for i = 1, 3 do
                        if math.abs(newTrans[i]-curTrans[i]) > 0.001 then
                            animPart.isMoving = true
                        end
                    end
                end

                if animPart.scaleMin ~= nil then
                    local curScale = {getScale(animPart.node)}
                    local newScale = Utils.getMovedLimitedValues(curScale, animPart.scaleMax, animPart.scaleMin, 3, animPart.moveTime, dt, not animPart.state)
                    setScale(animPart.node, unpack(newScale))
                    for i = 1, 3 do
                        if math.abs(newScale[i]-curScale[i]) > 0.001 then
                            animPart.isMoving = true
                        end
                    end
                end

                if animPart.permRotAxis ~= nil then
                    local spd = ((2*math.pi) / animPart.moveTime) * dt
                    local newRot = {0,0,0}
                    newRot[animPart.permRotAxis] = spd
                    rotate(animPart.node, unpack(newRot))
                    animPart.isMoving = animPart.state
                end

                if animPart.isMovingTool then
					if self.setMovingToolDirty ~= nil then
                    	self:setMovingToolDirty(animPart.node)
					end
                end
            end
        end
    end
end

function ToggleAnimatedParts:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	local spec = self[g_specName]
	self:clearActionEventsTable(spec.actionEvents)

	if isActiveForInput then
		for i, animPart in ipairs(spec.animParts) do
			local inputAction = InputAction[animPart.actionId]
			if inputAction then
				local actionEventId = nil

				if animPart.autoReturn then
					_, actionEventId = self:addActionEvent(spec.actionEvents, inputAction, self, ToggleAnimatedParts.actionTogglePart, false, false, true, true, i)
				else
					_, actionEventId = self:addActionEvent(spec.actionEvents, inputAction, self, ToggleAnimatedParts.actionTogglePart, true, false, false, true, i)
				end

				if actionEventId then
					if animPart.showHelp then
						local actionText = animPart.actionText
						if actionText then
							if g_i18n:hasText(actionText) then
								actionText = g_i18n:getText(actionText)
							end
							g_inputBinding:setActionEventText(actionEventId, actionText)
						end
						g_inputBinding:setActionEventTextVisibility(actionEventId, true)
					end
				end
			end
		end
	end
end

function ToggleAnimatedParts:onReadStream(streamId, connection)
	if connection:getIsServer() then
		local spec = self[g_specName]

		local numParts = streamReadUInt16(streamId)
		if numParts > 0 then
			for i = 1, numParts do
				local animPart = spec.animParts[i]
				if animPart then
					animPart.state = streamReadBool(streamId)
					animPart.isMoving = true
				end
			end
		end
	end
end

function ToggleAnimatedParts:onWriteStream(streamId, connection)
	if not connection:getIsServer() then
		local spec = self[g_specName]

		streamWriteUInt16(streamId, #spec.animParts)
		for i, animPart in ipairs(spec.animParts) do
			streamWriteBool(streamId, animPart.state)
    	end
	end
end

-- [Action Event Functions] *************************************************************************************************************************

function ToggleAnimatedParts:actionTogglePart(actionName, inputValue, callbackState)
	local spec = self[g_specName]
	local animPart = spec.animParts[callbackState]

	if animPart.autoReturn then
		if inputValue == 1 then
			animPart.state = true
		else
			animPart.state = false
		end
	else
		animPart.state = not animPart.state
	end

	ToggleAnimEvent.sendEvent(self, animPart.index, animPart.state)
end

-- [ToggleAnimEvent] ##################################################################################################################################################################################

ToggleAnimEvent = {}
ToggleAnimEvent_mt = Class(ToggleAnimEvent, Event)
InitEventClass(ToggleAnimEvent, "ToggleAnimEvent")

function ToggleAnimEvent:emptyNew()
    local self = Event:new(ToggleAnimEvent_mt)
    return self
end

function ToggleAnimEvent:new(object, index, state)
    local self = ToggleAnimEvent:emptyNew()
	if self then
       	self.object = object
       	self.index  = index
       	self.state  = state
	end
    return self
end

function ToggleAnimEvent:writeStream(streamId, connection)
  	NetworkUtil.writeNodeObject(streamId, self.object)
    streamWriteUInt16(streamId, self.index)
    streamWriteBool(streamId, self.state)
end

function ToggleAnimEvent:readStream(streamId, connection)
    self.object = NetworkUtil.readNodeObject(streamId)
    self.index  = streamReadUInt16(streamId)
    self.state 	= streamReadBool(streamId)

	self:run(connection)
end

function ToggleAnimEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(self, false, connection, self.object)
	end

	local spec = self.object[g_specName]
	local animPart = spec.animParts[self.index]

	animPart.state = self.state
	animPart.isMoving = true
end

function ToggleAnimEvent.sendEvent(vehicle, index, state, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
            g_server:broadcastEvent(ToggleAnimEvent:new(vehicle, index, state), nil, nil, vehicle)
        else
            g_client:getServerConnection():sendEvent(ToggleAnimEvent:new(vehicle, index, state))
        end
    end
end
